/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PromptProvider.h
 *    Provide a uniform interface for displaying error messages and questions.
 *
 * @author Ronny Stricker
 * @date   2011/08/31
 */

#ifndef _MIRA_PROMPTPROVIDER_H_
#define _MIRA_PROMPTPROVIDER_H_

#include <core/Credential.h>

#include <svn_auth.h>
#include <string>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

typedef svn_auth_ssl_server_cert_info_t ServerCertificate;

/**
 * @brief Abstract base class for handling repository login issues and
 * providing human readable error messages.
 */
class PromptProvider {

public:
	/**
	 * @brief Accept values used for server certificate acceptance.
	 */
	enum AcceptValue {
		Reject, Temporary, Always
	};

public:
	virtual ~PromptProvider() {}

	/**
	 * @brief request input.
	 */
	virtual std::string getInput( std::string const& message ) = 0;

	/**
	 * @brief request selection.
	 */
	virtual unsigned int getSelection( std::string const& message, unsigned int maxSelection, unsigned int defaultSelection = 0 ) = 0;

	/**
	 * @brief request login data.
	 * Function should provide login data in credential.
	 */
	virtual bool getLogin( Credential& credential, bool forceUserInput = false ) = 0;

	/**
	 * @brief request acceptance of server certificate.
	 * Ask user to accept the certificate provided in certificate.
	 */
	virtual AcceptValue acceptServerCertificate( ServerCertificate const& certificate ) = 0;

	/**
	 * @brief display given title and message.
	 */
	virtual void showMessage( std::string const& title, std::string const& message ) = 0;

	/**
	 * @brief Display the given title and message and ask yes or no.
	 */
	virtual bool showYesNoMessage( std::string const& title, std::string const& message, bool defaultSelection = true ) = 0;

	/**
	 * @brief Display the given title and message and asks to select one of the given options.
	 */
	virtual unsigned int showSelectionMessage( std::string const& title, std::string const& message, std::vector<std::string> const& options, unsigned int defaultSelection = 0, bool showOptions = true ) = 0;

	/**
	 * @brief Display the given title and message and asks to insert text.
	 */
	virtual std::string showInputMessage( std::string const& title, std::string const& message, std::string const& prefix = "" ) = 0;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
