/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageGroup.h
 *    Implementation of package group.
 *
 * @author Ronny Stricker
 * @date   2011/08/31
 */

#include <utils/Path.h>
#include <utils/EnumToFlags.h>

#include <factory/Factory.h>
#include <serialization/XMLSerializer.h>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/vector>

#ifndef _MIRA_PACKAGEGROUP_H_
#define _MIRA_PACKAGEGROUP_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class Package;
class Database;

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Base class for package items.
 * A package group represents a group of real packages. Therefore, it
 * consists of a list of child items, a name and a description but
 * cannot be associated with a "real" package.
 */
class PackageGroup : virtual public Object
{
	MIRA_OBJECT( PackageGroup )

public:
	enum Type {
		UNSPECIFIED = 0x0000,   /// if the package type is unknown (only used
		                        /// during dependency resolving)
		ETC    = 0x0001,
		BINARY = 0x0002,
		SOURCE = 0x0004
	};

	MIRA_ENUM_TO_FLAGS_INCLASS( Type );

public:
	PackageGroup();

	PackageGroup( std::string const& name );

	virtual ~PackageGroup();

	/**
	 * @brief Delete all child packages and clear child list.
	 */
	void clearChildren();

	/**
	 * @brief Remove child with the given pointer
	 * The child will be removed from the list of children and deleted.
	 * @param child pointer of the child
	 */
	void removeChild( PackageGroup* child );

	/**
	 * @brief Add package to the package subtree.
	 * If no package with that name exists, the package is
	 * added as a child of this. If an identical package
	 * can be found, the packages are merged. If one single
	 * package with that name exists (which is not identical to other)
	 * a new PackageGroup is created and both packages are added.
	 * The PackageGroup takes ownership of the new PackageGroup.
	 * @param[in] other The package which should be added.
	 */
	virtual void add( PackageGroup* other );

	/**
	 * @brief Add package other as a direct child.
	 * Since no checks are performed this function may lead to an
	 * invalid tree structure.
 	 * The PackageGroup takes ownership of the new PackageGroup.
 	 * @param[in] other The package which should be added.
	 */
	virtual void addChild( PackageGroup* other );

	/**
	 * @brief Add all child packages of PackageGroup other.
	 * The function consumes PackageGroup other. Therefore, package
	 * other is invalid when the function returns.
	 * @param[in] other The package with the child packages to add
	 */
	virtual void merge( PackageGroup* other );

	/**
	 * @brief Get set of subpackages.
	 * @param[in] recursive Passes request to child packages if set to true.
	 */
	virtual std::set<Package*> getPackages( bool recursive = true ) const;

	/**
	 * @brief Get vector of subpackages with the specified name.
	 * @param[in] name Name used for comparison
	 * @param[in] recursive Passes request to child packages if set to true.
	 */
	virtual std::vector<Package*> findPackages( std::string const& name, bool recursive = true ) const;

	/**
	 * @brief Get package(Group) with the same properties as the given package(Group).
	 * The function recursively cycles through children and returns THE FIRST matching
	 * package.
	 * @param[in] other Pointer to package used for comparison.
	 * @param[in] ignoredFlags specify the flags which are "ignored" during comparison
	 * of packages (see isIdentical() )
	 */
	PackageGroup* getIdentical( PackageGroup const* other, Type ignoredFlags =  UNSPECIFIED  ) const;

	/**
	 * @brief Get vector of package(Groups) with the same properties as the given package(Group).
	 * The function recursivly cycles through children and returns EVERY matching
	 * package.
	 * @param[in] other Pointer to package used for comparison.
	 * @param[in] ignoredFlags specify the flags which are "ignored" during comparison
	 * of packages (see isIdentical() )
	 * @param[in|out] identical filled vector with identical package(groups)
	 */
	void getAllIdentical( PackageGroup const* other, std::vector<PackageGroup*>& identical,
			Type ignoredFlags =  UNSPECIFIED ) const;

	/**
	 * @brief Return PackageGroup with the given name.
	 * Only true PackageGroups will be returned.
	 * @param[in] name Name used as search criterion.
	 * @param[in] recursive Cycles recursively through children if set to true.
	 */
	virtual PackageGroup* findGroup( std::string const& name, bool recursive = true ) const;

	/**
	 * @brief Returns a deep copy of this.
	 */
	virtual PackageGroup* createCopy() const;

	/**
	 * @brief Returns a deep copy of this, which can be used as dependency.
	 * All packages (children) will be returned as dependency type.
	 */
	virtual PackageGroup* createDependency() const;

	/**
	 * @brief The reflection function.
	 */
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("name", mName, "Name of package group.");
		r.member("type", mType, "Type of the package.");
		r.member("parentPackage", mParentPackage, "The parent package.");
		r.member("subPackages", mSubPackages, "A vector of subpackages.");
		r.member("description", mDescription, "The description.");
	}

	/**
	 * @brief Operator provided to enable sorting of package(groups)
	 */
	bool operator<(const PackageGroup& other) const;

	/**
	 * @brief Returns true if the package(group) is a primary one.
	 * Always returns false since PackageGroups are no "real" packages, and
	 * therefore cannot be primary.
	 */
	virtual bool isPrimary() const;

	/**
	 * @brief Return Description up to the first occurrence of '.'
	 * @return
	 */
	std::string getShortDescription() const;

	/**
	 * @brief Return true if this and other are identical.
	 * Checks if the name of this and other is equal.
	 * Different comparissons may be applied in the subclasses.
	 * @param[in] other the package used for comparison
	 * @param[in] ignoredFlags specify the flags which are "ignored" during comparison
	 */
	virtual bool isIdentical( PackageGroup const* other, Type ignoredFlags =  UNSPECIFIED ) const;

	/**
	 * @brief Return true if other can be grouped with this.
	 * Checks if the name of this and other is equal.
	 */
	bool canBeGroupedWith( PackageGroup const* other) const;

	/**
	 * @brief Return vector of recursive primary sub packages.
	 * Return primary children of this and cycles recursively through all primary
	 * children.
	 */
	std::vector<Package*> getPrimarySubPackages() const;

	/**
	 * @brief Exchange the specified package with the new one.
	 * Nothing is done if oldAddress is no subpackage of this.
	 */
	void exchangePackage( PackageGroup* oldAddress, PackageGroup* newAddress );

	/**
	 * @brief Call selectStdRepo on every subpackage.
	 * @see Package::selectStdRepo() for further details.
	 */
	virtual void selectStdRepo( Database* db );

	std::string getTypeString() const;

public:
	std::string mName;              ///< Name of package group
	Type mType;                     ///< Type of the package
	PackageGroup* mParentPackage;   ///< the parent package
	std::string mDescription;       ///< Description
	std::vector<PackageGroup*> mSubPackages;	///< vector of subpackages
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
