/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageGroup.h
 *
 * @author Ronny Stricker
 * @date   2011/08/31
 */

#include <platform/Types.h>
#include <utils/Path.h>
#include <utils/EnumToFlags.h>

#include <factory/Factory.h>
#include <serialization/XMLSerializer.h>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/vector>

#include <core/PackageGroup.h>
#include <core/Url.h>

#ifndef _MIRA_PACKAGE_H_
#define _MIRA_PACKAGE_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class Database;

/**
 * @brief Represents a package.
 * In contrast to the PackageGroup this class represents a "real"
 * package. Usually, the Package should not have any subpackages
 * since it should be a group if more than one package with that
 * name exists.
 */
class Package : public PackageGroup
{
	MIRA_OBJECT( Package )

public:
	enum DevelopmentState {
		DEPRECATED   = 0,
		EXPERIMENTAL = 1,
		STABLE       = 2,
		RELEASE      = 3
	};

public:
	static std::string getDevelStateString( Package const& package )
	{
		switch ( package.mDevelState ) {
			case DEPRECATED:   return "DEPRECATED"; break;
			case EXPERIMENTAL: return "EXPERIMENTAL"; break;
			case STABLE:       return "STABLE"; break;
			case RELEASE:      return "RELEASE"; break;
			default: return "";
		}
	}

public:
	Package();

	Package( std::string const& name );

	virtual ~Package();

	/**
	 * @brief Merge package with given package.
	 * Add all repos of package other to this. Furthermore, mCurrentRepo and
	 * mLocalPath are copied from other.
	 */
	virtual void merge( PackageGroup* other );

	/**
	 * @brief Create a deep copy of this.
	 */
	virtual Package* createCopy() const;

	/**
	 * @brief Returns a deep copy of this, which can be used as dependency.
	 * All packages (children) will be returned as dependency type.
	 */
	virtual Package* createDependency() const;

	/**
	 * @brief Implementation used for package sorting.
	 */
	bool operator<(const Package& other) const;

	Package& operator=(Package const& other);

	/**
	 * @brief Check if package is equal to given package.
	 * If both packages are identical, the repository list
	 * can still be different.
	 * @param[in] other the package used for comparison
	 * @param[in] ignoredFlags specify the flags which are "ignored" during comparison
	 */
	virtual bool isIdentical( PackageGroup const* other, Type ignoredFlags = UNSPECIFIED ) const;

	template <class Reflector>
	void reflect( Reflector& r )
	{
		MIRA_REFLECT_BASE(r, PackageGroup);

		serialization::VersionType version = r.version(1, this);
		r.member("repos",mRepos,"");
		r.member("relativeRepoPath", mRepoSubPath, "");
		r.member("fileName", mFileName , "");
		r.member("Version",mVersion,"");
		r.member("DevelState",mDevelState,"",EXPERIMENTAL);
		r.member("dependencies", mDependencies, "");
		r.member("tags", mTags, "");
		r.member("author", mAuthor, "");
		r.member("localPath", mLocalPath, "");

		if (version >= 1)
			r.member("changelog", mChangeLog, "The changelog.");
	}

	/**
	 * @brief Return a list of repositories sorted by priority.
	 */
	std::vector<std::string> sortedRepos( Database* db ) const;

	/**
	 * @brief Return true if package is a binary one or a source package.
	 */
	virtual bool isPrimary() const;

	/**
	 * @brief Select the repository with the highes priority.
	 * mCurrentRepo is modified.
	 */
	virtual void selectStdRepo( Database* db );

public:

	struct Version {

		Version();

		Version( uint32 iMajor, uint32 iMinor, uint32 iPatch);

		std::string str() const;

		void fromString(const std::string& str);

		template <class Reflector>
		void reflect( Reflector& r )
		{
			serialization::VersionType v = r.version(2, this);
			if (v < 1) {
				// Version 0
				r.member("VMajor", mMajor,"");
				r.member("VMinor", mMinor,"");
				r.member("VBuild", mPatch,"");
			} else
			if (v == 1) {
				// Version 1
				r.member("VMajor", mMajor,"");
				r.member("VMinor", mMinor,"");
				r.member("VPatch", mPatch,"");
			} else {
				// Version 2
				r.member("VMajor",  mMajor,"");
				r.member("VMinor",  mMinor,"");
				r.member("VPatch",  mPatch,"");
				r.member("BuildId", mBuildId,"");
			}
		}

		/**
		 * @brief returns true if major and minar version do match.
		 */
		bool isBinaryCompatible( Version const& other ) const;

		bool isIndifferent() const;

		bool operator==( Version const& other ) const;

		bool operator!=( Version const& other ) const;

		bool operator<(Version const& other) const;

		bool operator>(Version const& other) const;

		bool operator>=(Version const& other) const;

		bool operator<=(Version const& other) const;

		uint32 mMajor;   ///< Version Major
		uint32 mMinor;   ///< Version Minor
		uint32 mPatch;   ///< Version Patch
		uint32 mBuildId; ///< Build number (should only used by binary packages!)
	};

	std::set<Url> mRepos;   ///< URLs of the different repository sources
	Url mCurrentRepo;       ///< the active repository for the package
	Url mRepoSubPath;       ///< sub path of the package related to the repository root
	Path mLocalPath;        ///< path were package is installed (or will be installed)
	std::string mFileName;  ///< filename of the corresponding package file

	Version mVersion;       ///< Version of the package

	DevelopmentState mDevelState; ///< Development state of the package

	std::list<Package*> mDependencies; ///< Dependencies of the package

	std::vector<std::string> mTags; ///< tags associated with the package

	std::string mAuthor;     ///< Author of the package
	std::string mChangeLog;  ///< The ChangeLog.
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
