/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MemoryMonitorView.C
 *    Implementation of MemoryMonitorView.h.
 *
 * @author Erik Einhorn
 * @date   2012/12/07
 */

#include <views/MemoryMonitorView.h>

#ifdef MIRA_LINUX

#include <serialization/Serialization.h>

#include <platform/Memory.h>

#include <QVBoxLayout>
#include <QHeaderView>
#include <QMenu>

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::MemoryMonitorView, mira::ViewPart);

///////////////////////////////////////////////////////////////////////////////

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MemoryMonitorView::MemoryMonitorView() : ui(NULL)
{
}

MemoryMonitorView::~MemoryMonitorView()
{
}

QWidget* MemoryMonitorView::createPartControl()
{
	ui = new UI(this);
	startTimer(5000);
	update();
	return ui;
}

void MemoryMonitorView::timerEvent(QTimerEvent *e)
{
	update();
}

inline void resetItem(QTableWidget* table, int i, int j, const QVariant& v, int alignment=Qt::AlignLeft, bool bold=false)
{
	QTableWidgetItem* item=table->item(i,j);
	if(item==NULL) {
		item = new QTableWidgetItem();
		item->setTextAlignment(alignment);
		table->setItem(i, j, item);
	}
	item->setData(Qt::DisplayRole, v);

	static bool haveFonts = false;
	static QFont boldFont;
	static QFont normalFont;
	if(!haveFonts) {
		normalFont = item->font();
		normalFont.setBold(false);
		boldFont = item->font();
		boldFont.setBold(true);
		haveFonts = true;
	}

	item->setFont(bold ? boldFont : normalFont);
}

void MemoryMonitorView::update()
{
	if (ui==NULL)
		return;

	if(!isVisible()) // do not update if we are not visible to save performance
		return;

	MemoryBlockInfoList blocks = getDetailedMemoryBlockUsage();

	// we need to disable sorting while inserting the data
	ui->table->setSortingEnabled(false);
	ui->table->setRowCount(blocks.size()+1);
	int i=0;
	MemoryBlockInfo total;
	foreach(const MemoryBlockInfo& block, blocks)
	{
		total.size += block.size;
		total.rss += block.rss;
		total.pss += block.pss;
		total.totalPrivate += block.totalPrivate;
		total.totalShared += block.totalShared;
		total.swap += block.swap;
		if(block.size<128) // show only those blocks that have a significant size
			continue;
		resetItem(ui->table,i,0, block.address.c_str(), Qt::AlignVCenter);
		resetItem(ui->table,i,1, (quint64)block.size,Qt::AlignRight | Qt::AlignVCenter);
		resetItem(ui->table,i,2, (quint64)block.rss,Qt::AlignRight | Qt::AlignVCenter);
		resetItem(ui->table,i,3, (quint64)block.pss,Qt::AlignRight | Qt::AlignVCenter);
		resetItem(ui->table,i,4, (quint64)block.totalPrivate,Qt::AlignRight | Qt::AlignVCenter);
		resetItem(ui->table,i,5, (quint64)block.totalShared,Qt::AlignRight | Qt::AlignVCenter);
		resetItem(ui->table,i,6, (quint64)block.swap,Qt::AlignRight | Qt::AlignVCenter);
		resetItem(ui->table,i,7, block.name.c_str(), Qt::AlignVCenter);
		++i;
	}

	// Store total sizes
	resetItem(ui->table,i,0, " Total", Qt::AlignVCenter,true);
	resetItem(ui->table,i,1, (quint64)total.size,Qt::AlignRight | Qt::AlignVCenter,true);
	resetItem(ui->table,i,2, (quint64)total.rss,Qt::AlignRight | Qt::AlignVCenter,true);
	resetItem(ui->table,i,3, (quint64)total.pss,Qt::AlignRight | Qt::AlignVCenter,true);
	resetItem(ui->table,i,4, (quint64)total.totalPrivate,Qt::AlignRight | Qt::AlignVCenter,true);
	resetItem(ui->table,i,5, (quint64)total.totalShared,Qt::AlignRight | Qt::AlignVCenter,true);
	resetItem(ui->table,i,6, (quint64)total.swap,Qt::AlignRight | Qt::AlignVCenter,true);
	resetItem(ui->table,i,7, "", Qt::AlignVCenter);

	ui->table->setRowCount(i+1);
	ui->table->setSortingEnabled(true);
}

///////////////////////////////////////////////////////////////////////////////

MemoryMonitorView::UI::UI(QWidget* parent) : QWidget(parent)
{
	setupUi();
}

void MemoryMonitorView::UI::setupUi()
{
	QVBoxLayout* verticalLayout = new QVBoxLayout(this);
	verticalLayout->setContentsMargins(0,3,0,0);
	table = new QTableWidget(this);
	table->setColumnCount(8);
	table->sortByColumn(3, Qt::DescendingOrder);
	table->setSortingEnabled(true);

	QFontMetrics fm(table->font());
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
	table->horizontalHeader()->resizeSection(0,fm.horizontalAdvance("0xABCDEF00ABCDEF00-0xABCDEF00ABCDEF00"));
	table->horizontalHeader()->resizeSection(1,fm.horizontalAdvance("Size (kB)MM"));
	table->horizontalHeader()->resizeSection(2,fm.horizontalAdvance("888888MM"));
	table->horizontalHeader()->resizeSection(3,fm.horizontalAdvance("888888MM"));
	table->horizontalHeader()->resizeSection(4,fm.horizontalAdvance("PrivateMM"));
	table->horizontalHeader()->resizeSection(5,fm.horizontalAdvance("SharedMM"));
	table->horizontalHeader()->resizeSection(6,fm.horizontalAdvance("888888MM"));
#else
	table->horizontalHeader()->resizeSection(0,fm.width("0xABCDEF00ABCDEF00-0xABCDEF00ABCDEF00"));
	table->horizontalHeader()->resizeSection(1,fm.width("Size (kB)MM"));
	table->horizontalHeader()->resizeSection(2,fm.width("888888MM"));
	table->horizontalHeader()->resizeSection(3,fm.width("888888MM"));
	table->horizontalHeader()->resizeSection(4,fm.width("PrivateMM"));
	table->horizontalHeader()->resizeSection(5,fm.width("SharedMM"));
	table->horizontalHeader()->resizeSection(6,fm.width("888888MM"));
#endif
	table->horizontalHeader()->setStretchLastSection(true);
	table->verticalHeader()->setDefaultSectionSize(fm.height()+6);
	table->setSelectionMode(QAbstractItemView::NoSelection);
	table->setAlternatingRowColors(true);

	QStringList l;
	l << "Address" << "Size (kB)" << "Rss" << "Pss" << "Private" << "Shared" << "Swap" << "Name";
	table->setHorizontalHeaderLabels(l);

	verticalLayout->addWidget(table);
}

///////////////////////////////////////////////////////////////////////////////

}

#endif // MIRA_LINUX
