/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkGraphView.h
 *    TODO Add description.
 *
 * @author Erik Einhorn
 * @date   02/15/2014
 */

#ifndef _MIRA_FRAMEWORKGRAPHVIEW_H_
#define _MIRA_FRAMEWORKGRAPHVIEW_H_

#include <QGraphicsView>
#include <QCheckBox>
#include <QLineEdit>

#include <QToolButton>
#include <QMovie>

#include <rcp/EditorPart.h>

#include <fw/FrameworkGraph.h>

#include <views/VisualizationGraphView.h>

class QGraphicsScene;
class QSvgRenderer;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class GraphViewUI;

class FrameworkGraphView : public EditorPart
{
Q_OBJECT
MIRA_META_OBJECT(FrameworkGraphView,
				("Category", "Information")
				("Name", "Framework Graph")
				("Description", "Shows all channels, authorities and units from all frameworks as connected graph."))

public:

	FrameworkGraphView();
	virtual ~FrameworkGraphView();

public:

	virtual QWidget* createPartControl();

private slots:

	void setButtonIcon(int frame);

	void setAutoScale(bool enabled);

private:
	friend class GraphViewUI;

	GraphViewUI* ui;
	QToolButton* mBtUpdate;
	QToolButton* mBtSaveGraph;
	QMovie* mBtUpdateAnimation;
	QCheckBox* mCbAutoScale;
	QCheckBox* mCbShowNamespaces;
	QCheckBox* mCbDependenciesOnly;
	QCheckBox* mCbElideEdgeLabels;
	QCheckBox* mCbShowHiddenUnits;
	QCheckBox* mCbShowFrameworks;
	QLineEdit* mLeFilterUnits;
	QLineEdit* mLeHide;
};

class GraphViewUI : public GraphView
{
	Q_OBJECT
public:

	GraphViewUI(FrameworkGraphView* view, QWidget* parent);
	~GraphViewUI();

private:

	// define a special unknown status value, this is somewhat hackish but should do the trick
	#define STATUS_UNKNOWN ((Status::StatusMode)(Status::ERROR+1))

	struct StatusMixin {
		StatusMixin() : status(STATUS_UNKNOWN) {}
		Status::StatusMode status;
	};

	struct ChannelMixin {
		ChannelMixin() : lastupdate(0) {}
		uint64 lastupdate;
	};

	typedef FrameworkGraph<StatusMixin,void,ChannelMixin> Graph;

private:

	template <typename Container>
	bool hideItem(const Container& c, const std::string& name);


	void dotAddChannel(std::ostream& os, Graph::ChannelNodePtr c);
	void dotAddAuthority(std::ostream& os, Graph::AuthorityNodePtr a);
	bool dotAddNamespace(std::ostream& os, Graph::NamespaceNodePtr n, int level, int& id,
	                     const std::set<Graph::AuthorityNodePtr>& visibleAuthorities,
	                     const std::set<Graph::ChannelNodePtr>& visibleChannels);

private:

	void discoverGraphThread();
	void graphHasChangedCb(); // called within discoverGraph thread
	void generateDotFromGraph(std::ostream& os);

signals:

	void startDiscoverGraph();
	void finishedDiscoverGraph();
	void graphHasChanged();

private slots:

	void onStartDiscoverGraph();
	void onFinishedDiscoverGraph();
	void onGraphHasChanged();

public slots:

	void updateGraph();

private:

	FrameworkGraphView* mView;

	boost::thread mDiscoverThread;
	Graph mGraph;

	Time mLastChangedGraphUpdate;
};



///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
