/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CollisionTest.C
 *    Implementation of CollisionTest.h.
 *
 * @author Tim Langner
 * @date   2011/05/09
 */

#include <array>

#include <fstream>

#include <opencv2/imgproc/imgproc.hpp>

#include <math/Saturate.h>
#include <model/CollisionTest.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

void CollisionTest::initialize(const Footprint& footprint)
{
	assert(mSegments > 0);
	assert(mCellSize > 0.0f);

	mShapeLUT.clear();
	mShapeLUTBB.clear();

	// divide 360 degree in nr. segments
	mSegmentStep = two_pi<float>()/mSegments;

	// 2x the outer radius of the footprint defines the maximum
	// extent of the footprint for each rotation
	// so use it as the size of our lut
	float shapeRadius = footprint.getOuterRadius();
	mShapeDimension = (int)(2*shapeRadius/mCellSize)+1;

	// make sure that dimension is odd
	mShapeDimension |= 0x01;

	std::vector<cv::Point> rotatedHull;

	mShapeLUT.resize(mSegments+1);
	mShapeLUTBB.resize(mSegments+1);

	int x0 = mShapeDimension>>1;
	int y0 = mShapeDimension>>1;
	for (uint32 i=0; i<=mSegments; ++i)
	{
		Img8U1 bitmap(mShapeDimension, mShapeDimension);
		bitmap = 0;

		// calculate the orientation of the current segment and
		// do some pre-calculations
		// orientation of current segment is:
		// -(segment number * degree per segment - 180°)
		// the inversion is needed because the footprint is given in
		// world coordinates so rotation is done counterclockwise
		// but for the lut we need it to be clockwise
		float phi = -(i*mSegmentStep-pi<float>());
		float cos_phi=cos(phi);
		float sin_phi=sin(phi);

		// for each polygon of the footprint...
		foreach(const Polygon2f& polygon, footprint)
		{
			// ...rotate each point of the polygon and put it in the rotated hull
			rotatedHull.resize(polygon.size());
			for ( size_t j=0; j<polygon.size(); j++ )
			{
				float px = polygon[j].x()*cos_phi + polygon[j].y()*sin_phi;
				float py = polygon[j].x()*sin_phi - polygon[j].y()*cos_phi;
				int x = (int)(px / mCellSize) + x0;
				int y = (int)(py / mCellSize) + y0;
				rotatedHull[j].x = x;
				rotatedHull[j].y = y;
			}
			// make a C array out of our rotated hull vector
			// since opencv is refusing to use STL
			const cv::Point* h[1] = {rotatedHull.data()};
			int n[1] = {int(rotatedHull.size())};
			// draw the filled polygon of the rotated shape (current segment)
			cv::fillPoly((cv::Mat&)bitmap, h, n, 1, CV_RGB(255, 255, 255));
		}
		// we need to flip the img around x-Axis since the footprint is
		// given in the world coordinate system where x points right and
		// y points upward. however in images we have a coordinate
		// system where x points right and y points downward.
		cv::flip(bitmap, mShapeLUT[i], 0);

		// calculate bounding box for shape
		for (int y = 0; y < mShapeDimension; ++y) {
			const uint8* lutData = mShapeLUT[i].data(y);
			for (int x = 0; x < mShapeDimension; ++x) {
				if (lutData[x] == 255)
					mShapeLUTBB[i] |= Point2i(x,y);
			}
		}
	}
}

void CollisionTest::initialize(const Footprint& footprint, float cellSize)
{
	mCellSize = cellSize;
	initialize(footprint);
}

void CollisionTest::initialize(const Footprint& footprint, float cellSize, uint32 segments)
{
	mCellSize = cellSize;
	mSegments = segments;
	initialize(footprint);
}

bool CollisionTest::testCollision(const Img8U1& map,
                                  const Point2i& p, float phi) const
{
	Point2i offset;
	Rect2i overlap;
	std::tie(offset, overlap) = calculateOffsetAndOverlap(map.size(), p, phi);
	if (!overlap.isValid())
		return false;

	const Img8U1& shape = getShape(phi);

	// test for each pixel in the shape lut...
	for (int y = overlap.y0(); y < overlap.y1(); y++)
	{
		const uint8* shapeData = shape.data(y);
		const unsigned char* mapData = map.data(y + offset.y());
		for (int x = overlap.x0(); x < overlap.x1(); x++)
			// ... if we have a pixel in the shape (255) and
			// a pixel with a value > obstacle threshold in the map
			// the shape collides with an obstacle
			if (shapeData[x] == 255 && mapData[x+offset.x()] > mObstacleThreshold)
				return true;
	}
	// no collision found
	return false;
}

float CollisionTest::distanceToObstacle(const Img32F1& distanceMap,
                                        const Point2i& p, float phi,
                                        Point2i* closestModelPoint) const
{
	Point2i offset;
	Rect2i overlap;
	std::tie(offset, overlap) = calculateOffsetAndOverlap(distanceMap.size(), p, phi);
	if (!overlap.isValid())
		return FLT_MAX;

	const Img8U1& shape = getShape(phi);

	// test for each pixel in the shape lut...
	float minDistance = FLT_MAX;
	for (int y = overlap.y0(); y < overlap.y1(); y++)
	{
		const uint8* shapeData = shape.data(y);
		const float* mapData = (float*) distanceMap.data(y + offset.y());
		for (int x = overlap.x0(); x < overlap.x1(); x++)
		{
			// ... if a pixel in the shape is set
			if (shapeData[x] == 255)
			{
				// if distance of that pixel in the map is 0
				// we collide
				if (mapData[x + offset.x()] == 0.0f) {
					if (closestModelPoint)
						*closestModelPoint = Point2i(x + offset.x(), y + offset.y());
					return 0.0f;
				}
				// else check if its closer to an obstacle than the
				// current minimum distance
				if (minDistance > mapData[x + offset.x()]) {
					if (closestModelPoint)
						*closestModelPoint = Point2i(x + offset.x(), y + offset.y());
					minDistance = mapData[x + offset.x()];
				}
			}
		}
	}
	return minDistance;
}

boost::optional<Point2i> CollisionTest::obstaclePosition(
                                          const Img32F1& distanceMap, const Img<int32>& labelMap,
                                          const Point2i& closestModelPoint,
                                          std::map<int, Point2i>* cache)
{
	const float* mapData = (float*) distanceMap.data(closestModelPoint.y());
	const int32* labelData = (int32*) labelMap.data(closestModelPoint.y());
	int label = labelData[closestModelPoint.x()];

	if (label == 0)
		return boost::optional<Point2i>();

	// if obstacle point for this label already known (from previous search), just return it
	if (cache) {
		std::map<int, Point2i>::const_iterator f = cache->find(label);
		if (f != cache->end())
			return f->second;
	}

	assert(distanceMap.size() == labelMap.size());

	// check the search start position
	if (mapData[closestModelPoint.x()] == 0.0f) {
		if (cache)
			(*cache)[labelData[closestModelPoint.x()]] = closestModelPoint;
		return closestModelPoint;
	}

	// search incrementally outwards from the found closest point
	for (int radius = 1; radius < std::max(distanceMap.width(), distanceMap.height()); ++radius) {
		int x0 = std::max(closestModelPoint.x() - radius, 0);
		int x1 = std::min(closestModelPoint.x() + radius, distanceMap.width() - 1);
		int y0 = std::max(closestModelPoint.y() - radius, 0);
		int y1 = std::min(closestModelPoint.y() + radius, distanceMap.height() - 1);

		mapData = (float*) distanceMap.data(y0);
		labelData = (int32*) labelMap.data(y0);
		for (int x = x0; x <= x1; ++x) {
			if (mapData[x] == 0.0f) {       // found an obstacle point, memorize
				Point2i pos(x, y0);
				if (cache)
					(*cache)[labelData[x]] = pos;
				if (labelData[x] == label)  // if it is the obstacle we were looking for, return it
					return pos;
			}
		}
		mapData = (float*) distanceMap.data(y1);
		labelData = (int32*) labelMap.data(y1);
		for (int x = x0; x <= x1; ++x) {
			if (mapData[x] == 0.0f) {       // found an obstacle point, memorize
				Point2i pos(x, y1);
				if (cache)
					(*cache)[labelData[x]] = pos;
				if (labelData[x] == label)  // if it is the obstacle we were looking for, return it
					return pos;
			}
		}

		for (int y = y0 + 1; y < y1; ++y) {
			mapData = (float*) distanceMap.data(y);
			labelData = (int32*) labelMap.data(y);
			if (mapData[x0] == 0.0f) {       // found an obstacle point, memorize
				Point2i pos(x0, y);
				if (cache)
					(*cache)[labelData[x0]] = pos;
				if (labelData[x0] == label)  // if it is the obstacle we were looking for, return it
					return pos;
			}
			if (mapData[x1] == 0.0f) {       // found an obstacle point, memorize
				Point2i pos(x1, y);
				if (cache)
					(*cache)[labelData[x1]] = pos;
				if (labelData[x1] == label)  // if it is the obstacle we were looking for, return it
					return pos;
			}
		}
	}

	// we should not get here
	MIRA_THROW(XRuntime, "Obstacle was not found");

	// we never get here
	return Point2i();
}

void CollisionTest::distanceTransform(const Img8U1& map, Img32F1& distanceMap) const
{
	Img8U1 thresholdImg;
	cv::threshold(map, thresholdImg, mObstacleThreshold, 255, cv::THRESH_BINARY_INV);
#if CV_MAJOR_VERSION >= 3 
	cv::distanceTransform(thresholdImg, distanceMap, cv::DIST_L2, 5);
#else
	cv::distanceTransform(thresholdImg, distanceMap, CV_DIST_L2, 5);
#endif
}

void CollisionTest::distanceTransform(const Img8U1& map, Img32F1& distanceMap,
                                      Img<int32>& labelMap) const
{
#if (((CV_MAJOR_VERSION == 2) && (CV_MINOR_VERSION >= 4)) || (CV_MAJOR_VERSION > 2))
	Img8U1 thresholdImg;
	cv::threshold(map, thresholdImg, mObstacleThreshold, 255, cv::THRESH_BINARY_INV);
#  if CV_MAJOR_VERSION >= 3 
	cv::distanceTransform(thresholdImg, distanceMap, labelMap, cv::DIST_L2, 5, cv::DIST_LABEL_PIXEL);
#  else
	cv::distanceTransform(thresholdImg, distanceMap, labelMap, CV_DIST_L2, 5, CV_DIST_LABEL_PIXEL);
#  endif
#else
	MIRA_THROW(XNotImplemented, "Computation of label map in distance transform needs OpenCV >= 2.4");
#endif
}

void CollisionTest::distanceTransformAndClearShape(const Img8U1& map, Img32F1& distanceMap,
                                                   const Point2i& p, float phi) const
{
	Img8U1 thresholdImg;

	cv::threshold(map, thresholdImg, mObstacleThreshold, 255, cv::THRESH_BINARY_INV);
	// Clear the map on the actual position with the shape.
	clearShape(thresholdImg, p, phi, 255);

#if CV_MAJOR_VERSION >= 3 
	cv::distanceTransform(thresholdImg, distanceMap, cv::DIST_L2, 5);
#else
	cv::distanceTransform(thresholdImg, distanceMap, CV_DIST_L2, 5);
#endif
}

void CollisionTest::distanceTransformAndClearShape(const Img8U1& map, Img32F1& distanceMap,
                                                   Img<int32>& labelMap,
                                                   const Point2i& p, float phi) const
{
#if (((CV_MAJOR_VERSION == 2) && (CV_MINOR_VERSION >= 4)) || (CV_MAJOR_VERSION > 2))
	Img8U1 thresholdImg;

	cv::threshold(map, thresholdImg, mObstacleThreshold, 255, cv::THRESH_BINARY_INV);
	// Clear the map on the actual position with the shape.
	clearShape(thresholdImg, p, phi, 255);

#  if CV_MAJOR_VERSION >= 3 
	cv::distanceTransform(thresholdImg, distanceMap, labelMap, cv::DIST_L2, 5, cv::DIST_LABEL_PIXEL);
#  else
	cv::distanceTransform(thresholdImg, distanceMap, labelMap, CV_DIST_L2, 5, CV_DIST_LABEL_PIXEL);
#  endif
#else
	MIRA_THROW(XNotImplemented, "Computation of label map in distance transform needs OpenCV >= 2.4");
#endif
}

const Img8U1& CollisionTest::getShape(float phi) const
{
	return mShapeLUT[calculateSegment(phi)];
}

const Rect2i& CollisionTest::getShapeBB(float phi) const
{
	return mShapeLUTBB[calculateSegment(phi)];
}

void CollisionTest::clearShape(Img8U1& map, const Point2i& p, float phi,
                               uint8 freeValue) const
{
	Point2i offset;
	Rect2i overlap;
	std::tie(offset, overlap) = calculateOffsetAndOverlap(map.size(), p, phi);
	if (!overlap.isValid())
		return;

	const Img8U1& shape = getShape(phi);

	// for each pixel in the shape that also
	// overlaps with the map check if pixel is set (255)
	// and set the pixel in the map to free space
	for (int y=overlap.y0(); y<overlap.y1(); y++)
	{
		const uint8* shapeData = (uint8*)shape.data(y);
		unsigned char* mapData = map.data(offset.y()+y);
		for (int x=overlap.x0(); x<overlap.x1(); x++)
		{
			if(shapeData[x]==255)
				mapData[offset.x()+x]=freeValue; // free space
		}
	}
}

void CollisionTest::decayShape(Img8U1& map, const Point2i& p, float phi,
                               float decayRate, float neutral,
                               bool decayObstacles, bool decayFreespace,
                               bool invertShape) const
{
	Point2i offset;
	Rect2i overlap;
	std::tie(offset, overlap) = calculateOffsetAndOverlap(map.size(), p, phi);
	if (!overlap.isValid())
		return;

	const Img8U1& shape = getShape(phi);

	const auto updateValue = [=](float value) {
		// always round towards neutral, otherwise at small decay rates map values may never change
		if ((decayObstacles && (value > neutral))) {
			return neutral + (value - neutral) * (1.0f - decayRate); // this will implicitly floor()
		}                                                            // on cast to uint8
		else if (decayFreespace && (value < neutral)) {
			return std::ceil(neutral + (value - neutral) * (1.0f - decayRate));
		}
		return value;
	};

	// in the overlap rectangle, check each pixel if it belongs to the shape
	for (int y=overlap.y0(); y<overlap.y1(); y++)
	{
		const uint8* shapeData = (uint8*)shape.data(y);
		unsigned char* mapData = map.data(offset.y()+y);
		for (int x=overlap.x0(); x<overlap.x1(); x++)
		{
			const auto inShape = (shapeData[x] == 255);
			if (inShape != invertShape) // xor
				mapData[offset.x() + x] = updateValue(mapData[offset.x() + x]);
		}
	}

	if (!invertShape)
		return;

	// for decay outside shape, also decay everything outside the overlap rectangle

	// above
	for (int y=0; y<offset.y()+overlap.y0(); y++)
	{
		unsigned char* mapData = map.data(y);
		for (int x = 0; x < map.width(); x++) {
			mapData[x] = updateValue(mapData[x]);
		}
	}

	// below
	for (int y=offset.y()+overlap.y1(); y<map.height(); y++)
	{
		unsigned char* mapData = map.data(y);
		for (int x = 0; x < map.width(); x++) {
			mapData[x] = updateValue(mapData[x]);
		}
	}

	for (int y=offset.y()+overlap.y0(); y<offset.y()+overlap.y1(); y++)
	{
		unsigned char* mapData = map.data(y);

		// left
		for (int x = 0; x < offset.x()+overlap.x0(); x++)
			mapData[x] = updateValue(mapData[x]);

		// right
		for (int x = offset.x()+overlap.x1(); x < map.width(); x++)
			mapData[x] = updateValue(mapData[x]);
	}
}

int CollisionTest::calculateSegment(float phi) const
{
	int segment = round(Anglef(phi + pi<float>()).rad() / mSegmentStep);
	return saturate(segment, 0, (int)mSegments);
}

std::pair<Point2i, Rect2i> CollisionTest::calculateOffsetAndOverlap(const Size2i& map,
                                                                    const Point2i& p,
                                                                    float phi) const
{
	const Rect2i& bb = getShapeBB(phi);
	if (!bb.isValid())
		return std::make_pair(Point2i(), Rect2i());

	Point2i offset(p.x() - mShapeDimension / 2,
	               p.y() - mShapeDimension / 2);

	Rect2i overlap;

	// clip the x and y start and end values so that we only
	// test the overlapping region of shape and map at p
	overlap.minCorner.y() = std::max(-offset.y(), bb.y0());
	overlap.maxCorner.y() = std::min(map.height()-offset.y(), bb.y1()+1);

	overlap.minCorner.x() = std::max(-offset.x(), bb.x0());
	overlap.maxCorner.x() = std::min(map.width()-offset.x(), bb.x1()+1);

	return std::make_pair(offset, overlap);
}

///////////////////////////////////////////////////////////////////////////////

}}
