/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Joint.h
 *    A joint representation.
 *
 * @author Tim Langner
 * @date   2011/11/25
 */

#ifndef _MIRA_JOINT_H_
#define _MIRA_JOINT_H_

#include <geometry/Point.h>
#include <transform/Pose.h>
#include <serialization/adapters/boost/optional.hpp>

#include <model/RigidModelExports.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

/**
 * A joint class that connects two links.
 * There are various types of joints available.
 */
class MIRA_RIGID_MODEL_EXPORT Joint
{
public:
	enum Type
	{
		FIXED,      ///< For a fixed joint all degrees of freedom are locked and its state is fixed.
		REVOLUTE,   ///< Can rotate about the given axis but is limited by upper and lower limit
		CONTINUOUS, ///< Can rotate about the given axis with no given limits
		PRISMATIC,  ///< Can move/slide along the given axis but is limited by upper and lower limit
		FLOATING,   ///< Can move freely in all degrees of freedom
	};

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Name", name, "Name of the link");
		r.member("Type", type, "The joint's type");
		r.member("Origin", origin, "Origin", Pose3());
		r.member("Axis", axis, "Axis", Point3f(1.0f, 0.0f, 0.0f));
		r.member("Parent", parent, "Parent link");
		r.member("Child", child, "Child link");
		r.member("LowerLimit", lowerLimit, "Lower limit (rad for revolute and "
		         "meter for prismatic)", -pi<float>());
		r.member("UpperLimit", upperLimit, "Upper limit (rad for revolute and "
		         "meter for prismatic)", pi<float>());
	}

	std::string name;   ///< Name of the joint
	Type type;          ///< The joint's type
	Pose3 origin;       ///< Specifies the transform from child link to parent link
	Point3f axis;       ///< Axis of rotation for REVOLUTE and CONTINUOUS joints, axis of translation for PRISMATIC joints
	std::string parent; ///< Parent link
	std::string child;  ///< Child link
	float lowerLimit;   ///< Lower limit for REVOLUTE (rad) and PRISMATIC (meter) joints
	float upperLimit;   ///< Upper limit for REVOLUTE (rad) and PRISMATIC (meter) joints
};

/// Pointer to a joint
typedef boost::shared_ptr<Joint> JointPtr;

///////////////////////////////////////////////////////////////////////////////

}}

#endif 
