/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationGraphView.C
 *    .
 *
 * @author Erik Einhorn, Ronny Stricker
 * @date   2014/03/12
 */

#include <views/VisualizationGraphView.h>

#include <QMenu>
#include <QPen>
#include <QBoxLayout>
#include <QFileDialog>
#include <QScrollBar>
#include <QToolButton>
#include <QCheckBox>

#include <visualization/VisualizationGraph.h>
#include <visualization/VisualizationDotGraph.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

VisualizationGraphView::VisualizationGraphView() :
	mView(NULL), mScene(NULL), mGraphItem(NULL), mSvgRenderer(NULL),
	mBtSaveGraph(NULL), mCbAutoScale(NULL)
{
	setUpdateInterval(40);
}

void VisualizationGraphView::setupVisualizations()
{
	// initialize all existing visualizations
	foreach(Visualization* vis, getVisualizations())
		vis->init(this);
}

void VisualizationGraphView::addVisualization(Visualization* vis)
{
	VisualizationGraph<std::string>* v = dynamic_cast<VisualizationGraph<std::string>*>(vis);
	assert(v!=NULL);
	v->init(this);
	VisualizationView::addVisualization(vis);
}

const Class& VisualizationGraphView::supportedVisualizationClass() const
{
	return VisualizationGraph<std::string>::CLASS();
}

const Class& VisualizationGraphView::defaultVisualizationClass() const
{
	return VisualizationDotGraph::CLASS();
}

void VisualizationGraphView::update(Duration dt)
{
	assert(mView!=NULL);
}

QWidget* VisualizationGraphView::createVisualizationPart()
{
	QWidget* w = new QWidget(this);

	QVBoxLayout* v = new QVBoxLayout();
	w->setLayout(v);

	QHBoxLayout* h = new QHBoxLayout();

	mView = new GraphView(this);

	connect( mView, SIGNAL(autoScaleChanged(bool)), this, SLOT(setAutoScale(bool)) );

	mBtSaveGraph = new QToolButton(w);
	mBtSaveGraph->setToolTip("Save graph as graphviz dot file");
	mBtSaveGraph->setIcon(QIcon(":/icons/FileSave.png"));
	connect(mBtSaveGraph, SIGNAL(clicked()), mView, SLOT(saveGraph()));
	h->addWidget(mBtSaveGraph);

	mCbAutoScale = new QCheckBox("Auto Scale", w);
	mCbAutoScale->setChecked(true);
	connect(mCbAutoScale, SIGNAL(clicked(bool)), mView, SLOT(setAutoScale(bool)));
	h->addWidget(mCbAutoScale);

	v->addLayout(h);
	v->addWidget(mView);

	mView->showText("No data...");

	startUpdateTimer();
	setupVisualizations();
	return w;
}

void VisualizationGraphView::setAutoScale(bool enabled)
{
	mCbAutoScale->setCheckState( enabled ? Qt::Checked : Qt::Unchecked );
	mView->setAutoScale( enabled );
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION( mira::VisualizationGraphView, mira::VisualizationView);
