/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseCovVisualization.h
 *    TODO Add description.
 *
 * @author Erik Einhorn
 * @date   2011/09/20
 */

#ifndef _MIRA_POSECOVVISUALIZATION_H_
#define _MIRA_POSECOVVISUALIZATION_H_

#include <serialization/Serialization.h>
#include <serialization/GetterSetter.h>

#include <CommonVisualizationsExports.h>

#include <transform/Pose.h>

#include <visualization/3d/AxesCovObject.h>
#include <visualization/3d/CovObject.h>

#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreSceneManager.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

enum TransformType
{
	POSE2,
	POSE2COV,
	POSE3,
	POSE3COV,
	INVALID,	// placeholder
};

template <typename Pose>
struct CovVisTrait
{
	static const TransformType type = INVALID;
	static const bool poseHasCovariance = false;
};

template<>
struct CovVisTrait<Pose2>
{
	static const TransformType type = POSE3;
	static const bool poseHasCovariance = false;
};

template<>
struct CovVisTrait<PoseCov2>
{
	static const TransformType type = POSE3COV;
	static const bool poseHasCovariance = true;
};

template<>
struct CovVisTrait<Pose3>
{
	static const TransformType type = POSE3;
	static const bool poseHasCovariance = false;
};

template<>
struct CovVisTrait<PoseCov3>
{
	static const TransformType type = POSE3COV;
	static const bool poseHasCovariance = true;
};

///////////////////////////////////////////////////////////////////////////////

class MIRA_GUICOMMONVISUALIZATIONS_EXPORT CovVisualization
{
public:

	CovVisualization()
	{
		mIsVisible = true;
		mSigma = 0.5f;
	}

	void setupScene(Ogre::SceneManager* sceneManager, AxesObject* axes)
	{
		mCovObject.reset(new CovObject(sceneManager, (Ogre::SceneNode*)(axes->getNode()->getParent())));
		mAxesCovObject.reset(new AxesCovObject(sceneManager, axes));

		setVisible(mIsVisible);
		mCovObject->setSigma(mSigma);
		mAxesCovObject->setSigma(mSigma);
	}

	void update(const Pose2& pose) {}
	void update(const Pose3& pose) {}

	void update(const PoseCov2& pose)
	{
		Eigen::Matrix2f tcov = pose.cov.block<2,2>(0,0);
		mCovObject->setCovariance(tcov);
		mCovObject->setPosition(Eigen::Vector3f(pose.x(), pose.y(), 0));

		mAxesCovObject->setCovariance(pose.cov(2,2));
	}

	void update(const PoseCov3& pose)
	{
		PoseCov3::YawPitchRollCovMatrixType tEulerCov;

		tEulerCov = pose.getYawPitchRollCov();
		//Eigen::Matrix3f tcov = pose.cov.template block<3,3>(0,0);
		Eigen::Matrix3f tcov = tEulerCov.block<3,3>(0,0);
		mCovObject->setCovariance(tcov);
		mCovObject->setPosition(pose.t);
		mAxesCovObject->setCovariance(tEulerCov.block<3,3>(3,3));
	}

	void setVisible(bool show) {
		mIsVisible = show;
		if(mAxesCovObject)
			mAxesCovObject->setVisible(show);
		if(mCovObject)
			mCovObject->setVisible(show);
	}

	bool isVisible() const {
		return mIsVisible;
	}

	void setSigma(float sigma) {
		mSigma = sigma;
		if(mAxesCovObject)
			mAxesCovObject->setSigma(sigma);
		if(mCovObject)
			mCovObject->setSigma(sigma);
	}

	float getSigma() const {
		return mSigma;
	}

	boost::shared_ptr<CovObject>     mCovObject;
	boost::shared_ptr<AxesCovObject> mAxesCovObject;

	bool mIsVisible;
	float mSigma;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
