###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Provide some macros for the MIRA package handling mechanism.
#
# During the configuration process of the build environment, all existing
# packages files will be searched and put into a PackageList.txt file.
#
# The following macros can only be called from a package file:
#   MIRA_PACKAGE_NAME
#   MIRA_DESCRIPTION
#   MIRA_TAGS
#   MIRA_AUTHOR
#   MIRA_VERSION
#   MIRA_MARK_AS_EXTERNAL_PACKAGE
#   MIRA_ADD_REQUIREMENT
#   MIRA_CHANGELOG
#
#   (Within these macros, the name of the package will automatically
#    determined based on the name of the package file.)
#
# Furthermore, the following macros are available.
#   MIRA_REQUIRE_PACKAGE
#   MIRA_OPTIONAL_PACKAGE
#   MIRA_RUNTIME_PACKAGE
#
# The following macros are internal ones and should not be used directly
#   MIRA_PACKAGE_TYPE
#   MIRA_BUILD_ID
#   MIRA_ADD_REQUIREMENT_FOR_PACKAGE
#
# Usage:
#   MIRA_PACKAGE_NAME(name)
#     This macro is optional you can use it the give the package a name
#     which is divergent to the package file name.
#
#   MIRA_DESCRIPTION(description)
#     Set the description of the current package.
#     The description will be stored during the build process in the
#     file ${CMAKE_BINARY_DIR}/description_${package}.txt.
#
#   MIRA_TAGS(tags)
#     Set some tags for the current package.
#
#   MIRA_AUTHOR(author)
#     Set the author information for the current package.
#
#   MIRA_PACKAGE_TYPE(type)
#     Set the type of the package used by mirapackage. Available values:
#     SOURCE|BINARY|ETC
#
#   MIRA_VERSION(major minor patch)
#     This macro will set the version information of the current package:
#       ${name}_VERSION    = ${major}.${minor}.${patch}
#       ${name}_APIVERSION = ${major}.${minor}
#
#   MIRA_BUILD_ID(buildid)
#     This is macro is only used by mirapackage to distinguish different
#     build of the same package version for binary compatibility.
#
#   MIRA_DEVELOPMENT_STATE(state)
#     Set the development state of the package. Available values:
#     EXPERIMENTAL|STABLE|RELEASE|DEPRECATED
#
#   MIRA_MARK_AS_EXTERNAL_PACKAGE()
#     This macro will mark the current package as a none-MIRA package. This
#     is useful/necessary for external libraries, which are handled as packages
#     during building (e.g. Ogre).
#
#   MIRA_ADD_REQUIREMENT(requirement)
#     This macro tells the make system, that ${requirement} (e.g. a
#     other package) is necessary for building/using the current package.
#     (This macro will be called from MIRA_REQUIRE_PACKAGE).
#
#   MIRA_CHANGELOG(ChangeLog)
#     Set the ChangeLog of the current package. 
#     The Changelog information will be stored during the build process in the
#     file ${CMAKE_BINARY_DIR}/changelog_${package}.txt.
#
#   MIRA_FIND_PACKAGE(name)
#     This macro tries to find the package within the PackageList file
#     If the package could be found the following variables will be set
#       PACKAGE_${name}_FOUND = "TRUE"
#       PACKAGE_${name}_PATH  = "absolute package path"
#     otherwise
#       PACKAGE_${name}_FOUND = "FALSE"
#
#   MIRA_REQUIRE_PACKAGE(name)
#     This macro will import all necessary dependencies for the specified
#     package ${name} for a new target.
#     An optional 2nd parameter can be used to specify a required
#     version of package ${name}. Note that you should only specify requirements
#     for major and minor version e.g.: 0.9 instead of 0.9.2
#
#   MIRA_OPTIONAL_PACKAGE(name)
#     This macro will import all necessary dependencies for the specified
#     package ${name} for a new target if the package could be found.
#     Variables will be set according to MIRA_FIND_PACKAGE.
#
#
# For each package the following global (CACHE INTERNAL) variables are used:
#   ${name}_AUTHOR
#   ${name}_MAJOR
#   ${name}_MINOR
#   ${name}_PATCH
#   ${name}_VERSION
#   ${name}_APIVERSION
#   ${name}_COMPONENTS
#   ${name}_REQUIREMENTS
#
# Author: Erik Einhorn, Christian Martin, Tim Langner, Ronny Stricker
#
###############################################################################

###############################################################################
# The following code is executed during the initial 'cmake' process,
# which configures the build environment.
# The code looks for packages files (*.package) and creates a PackageList.txt
# file in the root directory of the current cmake project.
###############################################################################

# MIRA_PATH comes from Prerequisites.cmake
macro(INDEX_ALL_PACKAGES)
	set(packageCount 0)
	file(WRITE "${CMAKE_SOURCE_DIR}/PackageList.txt" "")
	foreach(package_searchPath ${MIRA_PATH})
		set(packages "")
		message(STATUS "Looking for packages in ${package_searchPath}:")

		# Search for all package files and store them in PackageList.txt
		file(GLOB_RECURSE MIRA_ALL_PACKAGES "${package_searchPath}/*.package")
		foreach(package_fullPath ${MIRA_ALL_PACKAGES})
			get_filename_component(filename ${package_fullPath} NAME)
			get_filename_component(packageName ${package_fullPath} NAME_WE)
			set(packages "${packages} ${packageName}")

			if(${package_searchPath} STREQUAL ${CMAKE_SOURCE_DIR})
				set(packageIsLocal "true")
				MATH(EXPR packageCount "${packageCount}+1")
				set(descriptionFileName "${CMAKE_BINARY_DIR}/description_${packageName}.txt")
				file(REMOVE ${descriptionFileName})
			else()
				set(packageIsLocal "false")
			endif()

			file(APPEND "${CMAKE_SOURCE_DIR}/PackageList.txt"
				"${packageName} ${package_fullPath} ${packageIsLocal} false;")
		endforeach(package_fullPath)
		message("${packages}")
	endforeach(package_searchPath)
	SET_PROPERTY(GLOBAL PROPERTY PACKAGE_COUNT ${packageCount})
	SET_PROPERTY(GLOBAL PROPERTY CURRENT_PACKAGE_COUNT 0)
	#message(STATUS "Finished looking for packages")
endmacro(INDEX_ALL_PACKAGES)

###############################################################################

#
# An internal macro which determines the current package file name.
#
macro(__MIRA_GET_CURRENT_NAME name)
	get_filename_component(fileExt  ${CMAKE_CURRENT_LIST_FILE} EXT)
	get_filename_component(fileName ${CMAKE_CURRENT_LIST_FILE} NAME_WE)
	if(NOT ${fileExt} STREQUAL ".package")
		message(FATAL_ERROR "MIRA-Package macros can't be called from none package files.")
	endif()
	set(${name} ${fileName})
endmacro(__MIRA_GET_CURRENT_NAME)

#
# Internal macro for adding an internal package to the list of known packages.
#
macro(__MIRA_ADD_INTERNAL_PACKAGE name)
	list(FIND MIRA_INTERNAL_PACKAGE_LIST ${name} foundInInternal)
	list(FIND MIRA_EXTERNAL_PACKAGE_LIST ${name} foundInExternal)

	# Only add the package, if is not in the internal list and also not in the external list.
	if(${foundInInternal} EQUAL -1 AND ${foundInExternal} EQUAL -1)
		list(APPEND MIRA_INTERNAL_PACKAGE_LIST ${name})
		set(MIRA_INTERNAL_PACKAGE_LIST "${MIRA_INTERNAL_PACKAGE_LIST}" CACHE INTERNAL "" FORCE)

		# Added the default cpack component for this package
		set(${name}_COMPONENTS "${name}" CACHE INTERNAL "" FORCE)
	endif()
endmacro(__MIRA_ADD_INTERNAL_PACKAGE)

#
# Internal macro for adding an external package to the list of known packages.
#
macro(__MIRA_ADD_EXTERNAL_PACKAGE name)
	# Only add the package, if is not already in the list.
	list(FIND MIRA_EXTERNAL_PACKAGE_LIST ${name} found)
	if(${found} EQUAL -1)
		list(APPEND MIRA_EXTERNAL_PACKAGE_LIST ${name})
		set(MIRA_EXTERNAL_PACKAGE_LIST "${MIRA_EXTERNAL_PACKAGE_LIST}" CACHE INTERNAL "" FORCE)
	endif()

	# Ensure, that the external package is not in MIRA_INTERNAL_PACKAGE_LIST
	list(REMOVE_ITEM MIRA_INTERNAL_PACKAGE_LIST ${name})
	set(MIRA_INTERNAL_PACKAGE_LIST "${MIRA_INTERNAL_PACKAGE_LIST}" CACHE INTERNAL "" FORCE)
endmacro(__MIRA_ADD_EXTERNAL_PACKAGE)

#
# Internal macro which determine if the package is within the local project
#
macro(__MIRA_IS_LOCAL_PACKAGE name isLocal)

	unset(${isLocal})

	file(READ "${CMAKE_SOURCE_DIR}/PackageList.txt" allPackages)
	foreach(package_searchPath ${allPackages})
		SEPARATE_ARGUMENTS(package_searchPath)
		list(GET package_searchPath 0 packageName)
		string(COMPARE EQUAL "${packageName}" "${name}" found)
		if(found)
			list(GET package_searchPath 1 packageFullPath)
			list(GET package_searchPath 2 packageIsLocal)
			set(${isLocal} ${packageIsLocal})
			break()
		endif(found)
	endforeach(package_searchPath)

endmacro(__MIRA_IS_LOCAL_PACKAGE)

###############################################################################

macro(MIRA_PACKAGE_NAME name)
	# do nothing, since this macro is only used by mirapackage
endmacro(MIRA_PACKAGE_NAME)

###############################################################################

macro(MIRA_DESCRIPTION)

	set(name "")
	__MIRA_GET_CURRENT_NAME(name)
	__MIRA_ADD_INTERNAL_PACKAGE(${name})

	# Only store the description in file, if this is a local package
	__MIRA_IS_LOCAL_PACKAGE(${name} packageIsLocal)
	if(packageIsLocal)
		set(descriptionFileName "${CMAKE_BINARY_DIR}/description_${name}.txt")
		if (NOT EXISTS ${descriptionFileName})		
			GET_PROPERTY(packageCount GLOBAL PROPERTY PACKAGE_COUNT)
			GET_PROPERTY(currentCount GLOBAL PROPERTY CURRENT_PACKAGE_COUNT)
			MATH(EXPR currentCount "${currentCount}+1")
			message(STATUS "Processing ${name} (${currentCount}/${packageCount})")
			SET_PROPERTY(GLOBAL PROPERTY CURRENT_PACKAGE_COUNT ${currentCount})

			# Store description in a file.

			file(WRITE "${descriptionFileName}" "")
			foreach(descrLine ${ARGV})
				# replace all variants of <br> <br/> with \n
				string(REGEX REPLACE "<[bB][rR]/*>" "\n" descrLine "${descrLine}")

				# remove all html tags
				string(REGEX REPLACE "<[^>]+>" "" descrLine "${descrLine}")

				# add to file
				file(APPEND "${descriptionFileName}" "${descrLine}")
			endforeach()
		endif()
	endif()

endmacro(MIRA_DESCRIPTION)

###############################################################################

macro(MIRA_TAGS tags)

	set(name "")
	__MIRA_GET_CURRENT_NAME(name)
	__MIRA_ADD_INTERNAL_PACKAGE(${name})

	set(${name}_TAGS ${tags} CACHE INTERNAL "" FORCE)

endmacro(MIRA_TAGS)

###############################################################################

macro(MIRA_AUTHOR author)

	set(name "")
	__MIRA_GET_CURRENT_NAME(name)
	__MIRA_ADD_INTERNAL_PACKAGE(${name})

	set(${name}_AUTHOR ${author})

endmacro(MIRA_AUTHOR)

###############################################################################

macro(MIRA_PACKAGE_TYPE type)

	string(TOUPPER "${type}" upperType)
	string(REGEX REPLACE "SOURCE|BINARY|ETC" "" unknownType "${upperType}")
	if(unknownType)
		message(FATAL_ERROR "Package type ${unknownType} is not known")
	endif()

endmacro(MIRA_PACKAGE_TYPE)

###############################################################################

macro(MIRA_VERSION major minor patch)

	set(name "")
	__MIRA_GET_CURRENT_NAME(name)
	__MIRA_ADD_INTERNAL_PACKAGE(${name})

	set(${name}_MAJOR ${major} CACHE INTERNAL "" FORCE)
	set(${name}_MINOR ${minor} CACHE INTERNAL "" FORCE)
	set(${name}_PATCH ${patch} CACHE INTERNAL "" FORCE)

	set(${name}_VERSION    ${major}.${minor}.${patch} CACHE INTERNAL "" FORCE)
	set(${name}_APIVERSION ${major}.${minor}          CACHE INTERNAL "" FORCE)

endmacro(MIRA_VERSION)

###############################################################################

macro(MIRA_BUILD_ID buildid)
	# do nothing, since this macro is only used by mirapackage
endmacro(MIRA_BUILD_ID)

###############################################################################

macro(MIRA_DEVELOPMENT_STATE state)

	string(TOUPPER "${state}" upperState)
	string(REGEX REPLACE "EXPERIMENTAL|STABLE|RELEASE|DEPRECATED" "" unknownState "${upperState}")
	if(unknownState)
		message(FATAL_ERROR "Package state ${unknownState} is not known!")
	endif()

endmacro(MIRA_DEVELOPMENT_STATE)

###############################################################################

macro(MIRA_MARK_AS_EXTERNAL_PACKAGE)

	set(name "")
	__MIRA_GET_CURRENT_NAME(name)
	__MIRA_ADD_EXTERNAL_PACKAGE(${name})

endmacro()

###############################################################################

macro(MIRA_ADD_REQUIREMENT requires)

	set(name "")
	__MIRA_GET_CURRENT_NAME(name)
	__MIRA_ADD_INTERNAL_PACKAGE(${name})

	# Never add a package itself to its requirements.
	if(NOT ${name} STREQUAL ${requires})
		list(FIND ${name}_REQUIREMENTS ${requires} found)
		if(${found} EQUAL -1)
			list(APPEND ${name}_REQUIREMENTS ${requires})
			set(${name}_REQUIREMENTS "${${name}_REQUIREMENTS}" CACHE INTERNAL "" FORCE)
		endif()
	endif()

endmacro(MIRA_ADD_REQUIREMENT)

###############################################################################

macro(MIRA_ADD_REQUIREMENT_FOR_PACKAGE name requires)

	# Never add a package itself to its requirements.
	if(NOT ${name} STREQUAL ${requires})
		list(FIND ${name}_REQUIREMENTS ${requires} found)
		if(${found} EQUAL -1)
			list(APPEND ${name}_REQUIREMENTS ${requires})
			set(${name}_REQUIREMENTS "${${name}_REQUIREMENTS}" CACHE INTERNAL "" FORCE)
		endif()
	endif()

endmacro(MIRA_ADD_REQUIREMENT_FOR_PACKAGE)

###############################################################################

macro(MIRA_CHANGELOG)
endmacro(MIRA_CHANGELOG)

###############################################################################

macro(MIRA_FIND_PACKAGE name)

	file(READ "${CMAKE_SOURCE_DIR}/PackageList.txt" allPackages)
	set(PACKAGE_${name}_FOUND "FALSE")
	foreach(package_searchPath ${allPackages})
		SEPARATE_ARGUMENTS(package_searchPath)
		list(GET package_searchPath 0 packageName)
		string(COMPARE EQUAL "${packageName}" "${name}" found)
		if(found)
			list(GET package_searchPath 1 packageFullPath)
			set(PACKAGE_${name}_FOUND "TRUE")
			set(PACKAGE_${name}_PATH ${packageFullPath})
			break()
		endif(found)
	endforeach(package_searchPath)

endmacro(MIRA_FIND_PACKAGE)

###############################################################################
macro(__MIRA_INCLUDE_PACKAGE name)
	if(NOT PACKAGE_${name}_INCLUDED)
		## Uncomment to enable timing measurements
		##MESSAGE(">>>>>> ${name}")
		##execute_process(COMMAND date +%s%N OUTPUT_VARIABLE  _BEGIN_TIME)
		
		set(PACKAGE_${name}_AUTOLINKLIBS_BEFORE ${MIRAAutoLinkLibraries})
		set(MIRAAutoLinkLibraries) # clear libs to record libraries of this package only

		# include the package
		include( ${PACKAGE_${name}_PATH} )
		set(PACKAGE_${name}_INCLUDED "TRUE")
		set(PACKAGE_${name}_AUTOLINKLIBS ${MIRAAutoLinkLibraries}) # store libraries of this package
		
		set(MIRAAutoLinkLibraries ${PACKAGE_${name}_AUTOLINKLIBS_BEFORE})
		list(APPEND MIRAAutoLinkLibraries ${PACKAGE_${name}_AUTOLINKLIBS})
		set(PACKAGE_${name}_AUTOLINKLIBS_BEFORE) #unset
		
		##execute_process(COMMAND date +%s%N OUTPUT_VARIABLE  _END_TIME)
		##math(EXPR _DIFF_TIME "(${_END_TIME}-${_BEGIN_TIME})/1000000")
		##MESSAGE("<<<<<< ${name}: ${_DIFF_TIME}")
		##MESSAGE("<<<<<< ${name}")
	else()
		list(APPEND MIRAAutoLinkLibraries ${PACKAGE_${name}_AUTOLINKLIBS})
	endif()
endmacro(__MIRA_INCLUDE_PACKAGE)



macro(MIRA_REQUIRE_PACKAGE name)

	# try to find package
	MIRA_FIND_PACKAGE( ${name} )
	if(NOT PACKAGE_${name}_FOUND)
		message(FATAL_ERROR "Cannot find MIRA package file '" ${name} "'.")
	endif()

	__MIRA_INCLUDE_PACKAGE( ${name} )

	# optional 2nd parameter that specifies the minimum required version
	set(requiredVersion ${ARGV1})

	# Get the name of the current processed makefile
	get_filename_component(fileName ${CMAKE_CURRENT_LIST_FILE} NAME_WE)

	# Do we have a version requirement for ${name} ?
	if(DEFINED requiredVersion)

		# The version requirement can only be checked, if the required package
		# provides version information.
		if(NOT DEFINED ${name}_VERSION)
			message(FATAL_ERROR "Package '${fileName}' has a version "
			        "requirement for package '${name}', but '${name}' has no "
			        "version information.")
		endif()

		# To check the required version, we have to distinguish
		# between two-component and three-component versions.
		# Only using VERSION_EQUAL will not work!
		if (NOT ( 
		    ((${requiredVersion} MATCHES "[0-9]+\\.[0-9]+") AND
		     (${requiredVersion} VERSION_EQUAL ${name}_APIVERSION))
		   OR
		    ((${requiredVersion} MATCHES "[0-9]+\\.[0-9]+\\.[0-9]+") AND
		     (${requiredVersion} VERSION_EQUAL ${name}_VERSION))
		   )
		)
			message(FATAL_ERROR "Package '${fileName}' requires '${name}' "
			        " = ${requiredVersion}, but only version ${${name}_VERSION} "
			        "is available.")
		endif()
	endif()

	# If this macro was called from a package file, add the requirement.
	get_filename_component(fileExt  ${CMAKE_CURRENT_LIST_FILE} EXT)
	if(${fileExt} STREQUAL ".package")
#		message(STATUS "  Debug: Package '${fileName}' requires package '${name}'")
		MIRA_ADD_REQUIREMENT(${name})
	endif()

endmacro(MIRA_REQUIRE_PACKAGE)

###############################################################################

macro(MIRA_OPTIONAL_PACKAGE name)

	# try to find package
	MIRA_FIND_PACKAGE( ${name} )
	if(PACKAGE_${name}_FOUND)
		# include the package
		__MIRA_INCLUDE_PACKAGE(${name})

		# If this macro was called from a package file, add the requirement.
		get_filename_component(fileExt  ${CMAKE_CURRENT_LIST_FILE} EXT)
		get_filename_component(fileName ${CMAKE_CURRENT_LIST_FILE} NAME_WE)
		if(${fileExt} STREQUAL ".package")
#			message(STATUS "  Debug: Package '${fileName}' requires package '${name}'")
			MIRA_ADD_REQUIREMENT(${name})
		endif()
	ELSE()
		message(STATUS "Cannot find OPTIONAL MIRA package file '" ${name} "'.")
	endif()

endmacro(MIRA_OPTIONAL_PACKAGE)

###############################################################################

macro(MIRA_RUNTIME_PACKAGE package)

	set(name "")
	__MIRA_GET_CURRENT_NAME(name)

	# try to find package
	MIRA_FIND_PACKAGE( ${package} )
	if(PACKAGE_${package}_FOUND)
	ELSE()
		message(STATUS "Cannot find package file '${package}' that is required"
		        " by package '${name}' via logical dependency at runtime.")
	endif()

endmacro(MIRA_RUNTIME_PACKAGE)

###############################################################################

macro(MIRA_ADD_COMPONENT component)
	set(name "")
	__MIRA_GET_CURRENT_NAME(name)
	__MIRA_ADD_INTERNAL_PACKAGE(${name})

	list(FIND ${name}_COMPONENTS ${component} found)
	if(${found} EQUAL -1)
		list(APPEND ${name}_COMPONENTS ${component})
		set(${name}_COMPONENTS "${${name}_COMPONENTS}" CACHE INTERNAL "" FORCE)
	endif()
endmacro(MIRA_ADD_COMPONENT)

###############################################################################

include(${CMAKE_CURRENT_LIST_DIR}/PackageInstaller.cmake)
