###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Build a DEB package for a MIRA package using cmake and dpkg.
#
# This cmake script file needs the following variables to be defined:
#
#   MIRA_INSTALL_PACKAGE:          Name of the package
#   MIRA_INSTALL_VERSION:          Full version
#   MIRA_INSTALL_BUILD:            Build version
#   MIRA_INSTALL_AUTHOR:           Author(s) of the package
#   MIRA_INSTALL_DESCRIPTION_FILE: File name of the description
#   MIRA_INSTALL_CHANGELOG_FILE:   File name of the changelog
#   MIRA_INSTALL_COMPONENTS:       Components of this package
#   MIRA_INSTALL_REQUIREMENTS:     Requirements
#
#   MIRA_ROOT_DIR:                 Path to MIRA to locate some files.
#   BUILD_BINARY_DIR:              Current build environment
#
# Author: Christian Martin
#
###############################################################################

STRING(TOLOWER ${MIRA_INSTALL_PACKAGE} lowerPackageName)

SET(INSTALL_ROOT          "${BUILD_BINARY_DIR}/install_root")
SET(CMAKEINSTALL_PATH     "${INSTALL_ROOT}/${lowerPackageName}")
SET(DEB_CFG_PATH          "${INSTALL_ROOT}/${lowerPackageName}/debian")
SET(DEB_BUILD_CFG_PATH    "${INSTALL_ROOT}/${lowerPackageName}/DEBIAN")

SET(MIRA_INSTALL_PREFIX    "/opt/MIRA")

###############################################################################
# cleanup and re-create all directories

FILE(REMOVE_RECURSE ${INSTALL_ROOT})

MAKE_DIRECTORY(${CMAKEINSTALL_PATH})
MAKE_DIRECTORY(${DEB_CFG_PATH})

###############################################################################
# Install all cmake components for the current package

FOREACH(component ${MIRA_INSTALL_COMPONENTS})
	EXEC_PROGRAM(${CMAKE_COMMAND} ARGS
		-D "BUILD_TYPE=Release"
		-D "CMAKE_INSTALL_DO_STRIP=1"
		-D "CMAKE_INSTALL_COMPONENT=${component}"
		-D "CMAKE_INSTALL_PREFIX=${CMAKEINSTALL_PATH}/${MIRA_INSTALL_PREFIX}"
		-P "${BUILD_BINARY_DIR}/cmake_install.cmake"
	)
ENDFOREACH()

###############################################################################
# Check if there is at least one file installed. Otherwise dpkg will fail!

FILE(GLOB_RECURSE FileList "${CMAKEINSTALL_PATH}/${MIRA_INSTALL_PREFIX}/*")
LIST(LENGTH FileList FileListLen)
IF(${FileListLen} EQUAL 0)
	MESSAGE(STATUS "*** Skip building DEB for package ${MIRA_INSTALL_PACKAGE}, since there are no files to install.")
	RETURN()
ENDIF()

###############################################################################
# get some more system information

EXEC_PROGRAM(uname ARGS -i OUTPUT_VARIABLE DEB_ARCH)

###############################################################################
# create the DEBIAN/control file

SET(DEB_NAME     ${lowerPackageName})
SET(DEB_VERSION  ${MIRA_INSTALL_VERSION})
SET(DEB_BUILD    ${MIRA_INSTALL_BUILD})
SET(DEB_FILENAME mira-${lowerPackageName}_${MIRA_INSTALL_VERSION}-${MIRA_INSTALL_BUILD}_${DEB_ARCH}.deb)

SET(DEB_DESCR_SHORT "Package ${MIRA_INSTALL_PACKAGE}")

# read description from file
SET(DEB_DESCR_LONG "")
FILE(STRINGS ${MIRA_INSTALL_DESCRIPTION_FILE} descriptionList)
FOREACH(descr ${descriptionList})
	IF ("${DEB_DEPENDS}" STREQUAL "")
		SET(DEB_DESCR_LONG " ${descr}")
	ELSE()
		SET(DEB_DESCR_LONG "${DEB_DESCR_LONG}\n ${descr}")
	ENDIF()
ENDFOREACH()

# read changelog from file and reformat according the DEB changelog syntax
FILE(STRINGS ${MIRA_INSTALL_CHANGELOG_FILE} changelogList)
# TODO: Re-format changelog according the DEB changelog syntax

# requirements
SET(DEB_DEPENDS    "")
FOREACH(requirement ${MIRA_INSTALL_REQUIREMENTS})
	# replace a{op}b with "a ({op} b)"
	#STRING(REGEX REPLACE "(.*[^<>])=(.*)" "\\1 (>= \\2)"  requirement ${requirement})
	STRING(REGEX REPLACE "(.*)>=(.*)"     "\\1 (>= \\2)" requirement ${requirement})
	STRING(REGEX REPLACE "(.*)<=(.*)"     "\\1 (<= \\2)" requirement ${requirement})
	STRING(REGEX REPLACE "(.*)>([^<>].*)" "\\1 (>> \\2)"  requirement ${requirement})
	STRING(REGEX REPLACE "(.*)<([^<>].*)" "\\1 (<< \\2)"  requirement ${requirement})

	# Work-around for deb packages: replace '=' by '>='
	STRING(REGEX REPLACE "(.*[^<>])=(.*)" "\\1 (>= \\2)"  requirement ${requirement})

	STRING(TOLOWER ${requirement} lowerRequirement)

	IF ("${DEB_DEPENDS}" STREQUAL "")
		SET(DEB_DEPENDS "${lowerRequirement}")
	ELSE()
		SET(DEB_DEPENDS "${DEB_DEPENDS}, ${lowerRequirement}")
	ENDIF()
ENDFOREACH()

CONFIGURE_FILE(
	${MIRA_ROOT_DIR}/make/deb_control.in
	${DEB_CFG_PATH}/control @ONLY)

###############################################################################
# Use dpkg-shlibdeps to determine additional dependencies

FILE(GLOB shLibList "${CMAKEINSTALL_PATH}/${MIRA_INSTALL_PREFIX}/lib/*.so*")

LIST(LENGTH shLibList shLibListLen)
IF(NOT ${shLibListLen} EQUAL 0)

	# Call dpkg-shlibdeps to determine shared library dependencies
	EXEC_PROGRAM(dpkg-shlibdeps ${CMAKEINSTALL_PATH} ARGS opt/MIRA/lib/*.so*
		--ignore-missing-info RETURN_VALUE shlibdeps_return
	)
	IF(NOT ${shlibdeps_return} EQUAL 0)
		MESSAGE(FATAL_ERROR "Executing dpkg-shlibdeps failed.")
	ENDIF()

	# Read shared library dependencies from control/substvars
	FILE(READ ${DEB_CFG_PATH}/substvars shLibDeps)
	STRING(STRIP ${shLibDeps} shLibDeps)
	STRING(REPLACE "shlibs:Depends=" "" shLibDeps ${shLibDeps})
	SET(shLibDeps "${shLibDeps},")
ELSE()
	SET(shLibDeps "")
ENDIF()

# Replace ${shlibs:Depends} in control by content of control/substvars
FILE(READ ${DEB_CFG_PATH}/control debControl)
STRING(REPLACE "\${shlibs:Depends}" "${shLibDeps}" debControlTmp "${debControl}")
FILE(WRITE ${DEB_CFG_PATH}/control.tmp "${debControlTmp}")
FILE(RENAME ${DEB_CFG_PATH}/control.tmp ${DEB_CFG_PATH}/control)

###############################################################################
# Rename 'debian' to 'DEBIAN' to be able to use dpkg --build

FILE(RENAME ${DEB_CFG_PATH} ${DEB_BUILD_CFG_PATH})

###############################################################################
# build the DEB package using "dpkg --build"

MESSAGE(STATUS "*** Building DEB package ${DEB_FILENAME}")

EXEC_PROGRAM(dpkg ${INSTALL_ROOT} ARGS
	--build ${lowerPackageName} ./
	OUTPUT_VARIABLE dpkgbuild_output
	RETURN_VALUE dpkgbuild_return)

SET(DEBBUILD_LOG "${INSTALL_ROOT}/${MIRA_INSTALL_PACKAGE}.log")
FILE(WRITE ${DEBBUILD_LOG} "${dpkgbuild_output}")

IF(NOT ${dpkgbuild_return} EQUAL 0)
	MESSAGE(FATAL_ERROR "Failed to build DEB ${DEB_FILENAME}. "
	        "For more details see ${DEBBUILD_LOG}.")
ENDIF()

###############################################################################
# Move DEB file

FILE(RENAME 
		${INSTALL_ROOT}/${DEB_FILENAME}
		${CMAKE_CURRENT_BINARY_DIR}/${DEB_FILENAME})

###############################################################################
# Clean up

FILE(REMOVE_RECURSE ${INSTALL_ROOT})
