/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file QtUtils.C
 *    Implementation of QtUtils.h
 *
 * @author Tim Langner, Christof Schröter
 * @date   2011/02/06
 */

#include <widgets/QtUtils.h>

#include <algorithm>

#include <QImage>
#include <QFileDialog>
#include <QGraphicsItem>
#include <QMessageBox>
#include <QPainter>

#include <math/Saturate.h>
#include <serialization/Print.h>

namespace mira { namespace QtUtils {

///////////////////////////////////////////////////////////////////////////////

QVector<QRgb> makeColorTable()
{
	QVector<QRgb> colorTable(256);

	for (int i = 0; i < 256; ++i)
		colorTable[i]  = qRgb(i, i, i);

	return colorTable;
}

void setColorTable(QImage& qimg)
{
	static QVector<QRgb> colorTable = makeColorTable();
	qimg.setColorTable(colorTable);
}

QImage makeQImage(const Img8U1& img)
{
	QImage qimage(img.width(), img.height(), QImage::Format_Indexed8);

	for (int y = 0; y < img.height(); ++y)
		std::copy_n(img.data(y), img.width(), qimage.scanLine(y));

	setColorTable(qimage);
	return qimage;
}

QImage makeQImage(const Img<int8, 1>& img)
{
	QImage qimage(img.width(), img.height(), QImage::Format_Indexed8);

	for (int y = 0; y < img.height(); ++y) {
		const int8* imagePtr = reinterpret_cast<const int8*>(img.data(y));
		std::transform(imagePtr, imagePtr+img.width(),
		               qimage.scanLine(y),
		               [](int c){ return c + 128; });
	}

	setColorTable(qimage);
	return qimage;
}

QImage makeQImage(const Img8U3& img)
{
	QImage qimage(img.width(), img.height(), QImage::Format_RGB32);

	for (int y = 0; y < img.height(); ++y) {
		const uint8* imagePtr = img.data(y);
		uchar* qimagePtr      = qimage.scanLine(y);
		for (int x = 0; x < img.width(); ++x) {
			std::copy_n(imagePtr, 3, qimagePtr);
			qimagePtr[3] = 255; // with Format_RGB32, QImage::pixel() always returns
			                    // alpha=255/ff, just setting it here for consistency
			imagePtr  += 3;
			qimagePtr += 4;
		}
	}

	return qimage;
}

QImage makeQImage(const Img8U4& img)
{
	QImage qimage(img.width(), img.height(), QImage::Format_ARGB32);

	for (int y = 0; y < img.height(); y++)
		std::copy_n(reinterpret_cast<const int32*>(img.data(y)), img.width(),
		            reinterpret_cast<int32*>(qimage.scanLine(y)));

	return qimage;
}

template <typename T, typename U>
QImage makeQImage(const Img<T, 1>& img, T minVal, T maxVal)
{
	if (minVal >= maxVal)
		MIRA_THROW(XInvalidParameter, "makeQImage(): minVal >= maxVal: "
		                              << minVal << ", " << maxVal);

	QImage qimage(img.width(), img.height(), QImage::Format_Indexed8);

	const U range = maxVal - minVal;

	auto mapTo0to255 = [&minVal, &range](U c){
		return saturate<U>(255 * (c - minVal) / range, 0, 255); };

	for (int y = 0; y < img.height(); ++y) {
		const T* imagePtr = reinterpret_cast<const T*>(img.data(y));

		std::transform(imagePtr, imagePtr+img.width(),
		               qimage.scanLine(y), mapTo0to255);
	}

	setColorTable(qimage);
	return qimage;
}

QImage makeQImage(const Img16U1& img, uint16 minVal, uint16 maxVal)
{
	return makeQImage<uint16, int>(img, minVal, maxVal);
}

QImage makeQImage(const Img16S1& img, int16 minVal, int16 maxVal)
{
	return makeQImage<int16, int>(img, minVal, maxVal);
}

QImage makeQImage(const Img32F1& img, float minVal, float maxVal)
{
	return makeQImage<float, double>(img, minVal, maxVal);
}

QImage makeQImage(const Img64F1& img, double minVal, double maxVal)
{
	return makeQImage<double, double>(img, minVal, maxVal);
}

void signalUnsupportedFormat(const Img<>& img)
{
	MIRA_THROW(XNotImplemented, "makeQImage() not implemented for image format: "
	                            << print(img.format(), COMPACT_PRINT));
}

void signalScalingNotSupported(const Img<>& img)
{
	MIRA_THROW(XInvalidParameter,
	           "makeQImage() must NOT be used with minVal/maxVal for image format: "
	           << print(img.format(), COMPACT_PRINT));
}

void signalMissingScalingParams(const Img<>& img)
{
	MIRA_THROW(XInvalidParameter,
	           "makeQImage() must be used with minVal/maxVal for image format: "
	           << print(img.format(), COMPACT_PRINT));
}

QImage makeQImage(const Img<>& img)
{
	if(img.isEmpty())
		return QImage();

	QImage qimage;

	switch (img.depth())
	{
	case CV_8U:
		switch (img.channels())
		{
		case 1: qimage = makeQImage(Img8U1::convertFrom(img)); break;
		case 3: qimage = makeQImage(Img8U3::convertFrom(img)); break;
		case 4: qimage = makeQImage(Img8U4::convertFrom(img)); break;
		default: signalUnsupportedFormat(img);
		}
		break;
	case CV_8S:
		if (img.channels() == 1)
			qimage = makeQImage(Img<int8, 1>::convertFrom(img));
		else
			signalUnsupportedFormat(img);
		break;
	case CV_16U:
		if (img.channels() == 1)
			signalMissingScalingParams(img);
		else
			signalUnsupportedFormat(img);
		break;
	case CV_16S:
		if (img.channels() == 1)
			signalMissingScalingParams(img);
		else
			signalUnsupportedFormat(img);
		break;
	case CV_32F:
		if (img.channels() == 1)
			signalMissingScalingParams(img);
		else
			signalUnsupportedFormat(img);
		break;
	case CV_64F:
		if (img.channels() == 1)
			signalMissingScalingParams(img);
		else
			signalUnsupportedFormat(img);
		break;
	default: signalUnsupportedFormat(img);
	}

	return qimage;
}

QImage makeQImage(const Img<>& img, double minVal, double maxVal)
{
	if(img.isEmpty())
		return QImage();

	QImage qimage;

	switch (img.depth())
	{
	case CV_8U:
		switch (img.channels())
		{
		case 1: // fallthrough
		case 3: // fallthrough
		case 4: signalScalingNotSupported(img); break;
		default: signalUnsupportedFormat(img);
		}
		break;
	case CV_8S:
		if (img.channels() == 1)
			signalScalingNotSupported(img);
		else
			signalUnsupportedFormat(img);
		break;
	case CV_16U:
		if (img.channels() == 1)
			qimage = makeQImage(Img16U1::convertFrom(img), (uint16)floor(minVal), (uint16)ceil(maxVal));
		else
			signalUnsupportedFormat(img);
		break;
	case CV_16S:
		if (img.channels() == 1)
			qimage = makeQImage(Img16S1::convertFrom(img), (int16)floor(minVal), (int16)ceil(maxVal));
		else
			signalUnsupportedFormat(img);
		break;
	case CV_32F:
		if (img.channels() == 1)
			qimage = makeQImage(Img32F1::convertFrom(img), (float)minVal, (float)maxVal);
		else
			signalUnsupportedFormat(img);
		break;
	case CV_64F:
		if (img.channels() == 1)
			qimage = makeQImage(Img64F1::convertFrom(img), minVal, maxVal);
		else
			signalUnsupportedFormat(img);
		break;
	default: signalUnsupportedFormat(img);
	}

	return qimage;
}

void checkAlignment(const Img8U1& img)
{
	if (img.step() % 4)
		MIRA_THROW(XInvalidParameter, "asQImage(Img8U1&): "
		                              "image line offset (step) must be multiple of 4.");
}

// this internal function should only be called with T = Img8U1 or T = Img8U1 const
// (--> ref or const ref to Img8U1)
// but as it is not part of the public interface, the assumption is not explicitly
// checked to keep the code clean
template <typename T>
QImage asQImage(T& img)
{
	checkAlignment(img);

	QImage qimage(img.data(), img.width(), img.height(), QImage::Format_Indexed8);
	setColorTable(qimage);
	return qimage;
}

QImage asQImage(Img8U1& img)
{
	return asQImage<Img8U1>(img);
}

QImage asQImage(const Img8U1& img)
{
	return asQImage<Img8U1 const>(img);
}

QImage asQImage(Img8U4& img)
{
	return QImage(img.data(), img.width(), img.height(), QImage::Format_ARGB32);
}

QImage asQImage(const Img8U4& img)
{
	return QImage(img.data(), img.width(), img.height(), QImage::Format_ARGB32);
}

void fromQImage(const QImage& qimage, Img<>& img)
{
	size_t widthStep = img.step();
	int width  = qimage.width();
	int height = qimage.height();

	switch (qimage.format())
	{
	case QImage::Format_Indexed8:
	{
		img = Img8U1(width, height);
		for (int y = 0; y < height; y++)
		{
			const uchar* qimagePtr = qimage.scanLine(y);
			uint8* imagePtr        = img.data(y);
			memcpy(imagePtr, qimagePtr, width);
		}
		break;
	}
	case QImage::Format_RGB32:
	{
		img = Img8U3(width, height);
		for (int y = 0; y < height; y++)
		{
			const uchar* qimagePtr = qimage.scanLine(y);
			uint8* imagePtr        = img.data(y);
			for (int x = 0; x < width; x++)
			{
				imagePtr[0] = qimagePtr[0];
				imagePtr[1] = qimagePtr[1];
				imagePtr[2] = qimagePtr[2];
				qimagePtr += 4;
				imagePtr  += 3;
			}
		}
		break;
	}
	default:
		MIRA_THROW(XNotImplemented, "Conversion from QImage format "
			<< qimage.format() << " to Img<> not supported.");
		break;
	}
}

void setTransform(QGraphicsItem* item, const RigidTransform2f& t)
{
	item->setPos(t.x(), t.y());
#if (QT_VERSION < QT_VERSION_CHECK(4, 6, 1))
	//item->setTransform(QTransform(cos(t.phi()),-sin(t.phi()),sin(t.phi()),cos(t.phi()),0.0,0.0));
	// for the skeptic (maybe the sign is wrong)
	item->setTransform(QTransform().rotate(rad2deg(t.phi())));
#else
	item->setRotation(rad2deg(t.phi()));
#endif
}

QString getOpenFileName(QWidget* parent,
                        const QString& caption,
                        const QString& dir,
                        const QString& filter,
                        QString* selectedFilter,
                        QFileDialog::Options options) {
	return QFileDialog::getOpenFileName(parent, caption, dir, filter, selectedFilter,
	                                    options | QFileDialog::DontUseNativeDialog);
}

QString getSaveFileName(QWidget* parent,
                        const QString& caption,
                        const QString& dir,
                        const QString& filter,
                        QString* selectedFilter,
                        QFileDialog::Options options,
                        const QStringList& enforceExtension)
{
	QString currentDir = dir;
	while (true) {
		QString filename =  QFileDialog::getSaveFileName(parent, caption, currentDir, filter, selectedFilter,
		                                                 options | QFileDialog::DontUseNativeDialog);

		if (enforceExtension.isEmpty() || filename.isEmpty())
			return filename;

		const QFileInfo info(filename);
		const QString suffix(info.suffix());

		if (enforceExtension.contains(suffix, Qt::CaseInsensitive))
			return filename;

		// we reached here as a filename was selected but it does not match the required extension
		// we add the extension, have to check for file overwrite ourselves now

		filename.append("."+enforceExtension.first());

		const QFileInfo infoExt(filename);
		if (!infoExt.exists())
			return filename;

		if (QMessageBox::warning(parent, "Save as CSV",
		                         infoExt.fileName() + " already exists.\nDo you want to replace it?",
		                         QMessageBox::No | QMessageBox::Yes) == QMessageBox::Yes) {
			return filename;
		}

		// will start dialog at last selected directory for making a new choice
		currentDir = infoExt.path();
	}
}

QString getExistingDirectory(QWidget* parent,
                             const QString& caption,
                             const QString& dir,
                             QFileDialog::Options options)
{
	return QFileDialog::getExistingDirectory(parent, caption, dir,
	                                         options | QFileDialog::DontUseNativeDialog);
}

void initPainterFrom(QPainter& painter, const QWidget& widget)
{
	// taken from implementation of QWidget::initPainter(QPainter*)
	// which is doing the work for QPainter::initFrom(QPaintDevice*),
	// but is not public, and probably also deprecated

//	void QWidget::initPainter(QPainter *painter) const
//	{
//		const QPalette &pal = palette();
//		painter->d_func()->state->pen = QPen(pal.brush(foregroundRole()), 1);
//		painter->d_func()->state->bgBrush = pal.brush(backgroundRole());
//		QFont f(font(), const_cast<QWidget *>(this));
//		painter->d_func()->state->deviceFont = f;
//		painter->d_func()->state->font = f;
//	}

	const QPalette& pal = widget.palette();
	painter.setPen(QPen(pal.brush(widget.foregroundRole()), 1));
	painter.setBackground(pal.brush(widget.backgroundRole()));
	painter.setFont(widget.font());
}

///////////////////////////////////////////////////////////////////////////////

}}
