/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseTool2D.C
 *    Implementation of PoseTool2D.
 *
 * @author Christian Reuther
 * @date   2018/08/29
 */

#include <visualization/2d/PoseTool2D.h>

#include <QMouseEvent>
#include <QGraphicsView>
#include <QDebug>
#include <qmath.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PoseTool2D::PoseTool2D(float minPositionVariance, float minOrientationVariance) : mSelectionMode(Position), mIsValid(false),
	mMinPositionVariance(minPositionVariance), mMinOrientationVariance(minOrientationVariance)
{
	mMinPositionVariance = std::max<float>(0.01f, mMinPositionVariance);
	mItem = NULL;
	mLineXItem = NULL;
	mLineYItem = NULL;
	mCovXYItem = NULL;
	mCovPhiXItem = NULL;
	mCovPhiYItem = NULL;
}

PoseTool2D::~PoseTool2D() {
	auto site = this->getSite();
	if(!site || !site->getSceneManager())
		return;

	site->getSceneManager()->removeItem(mItem);
	if(mLineXItem) {
		mItem->removeFromGroup(mLineXItem);
		delete mLineXItem;
		mLineXItem = NULL;
	}
	if(mLineYItem) {
		mItem->removeFromGroup(mLineYItem);
		delete mLineYItem;
		mLineYItem = NULL;
	}
	if(mCovPhiXItem) {
		mItem->removeFromGroup(mCovPhiXItem);
		delete mCovPhiXItem;
		mCovPhiXItem = NULL;
	}
	if(mCovPhiYItem) {
		mItem->removeFromGroup(mCovPhiYItem);
		delete mCovPhiYItem;
		mCovPhiYItem = NULL;
	}

	// Do not delete mItem as the base class destructor does this.
}

///////////////////////////////////////////////////////////////////////////////

void PoseTool2D::onMousePressed(QMouseEvent* e) {
	// The right mouse button disables/deactivates the tool
	if(e->button() == Qt::RightButton) {
		mIsValid = false;
		mSelectionMode = Position;
		deactivate();
		return;
	}

	if(e->button() != Qt::LeftButton)
		return;

	mItem->setVisible(true);
	if(mSelectionMode == Position) {
		mLastPos = getSite()->getViewManager()->mapToScene(e->pos());
		mPose.x() = mLastPos.x();
		mPose.y() = mLastPos.y();
		mPose.cov.setZero(3,3);
		mPose.cov(0,0) = mMinPositionVariance;
		mPose.cov(1,1) = mMinPositionVariance;
		mPose.cov(2,2) = mMinOrientationVariance;
		mSelectionMode = PositionVariance;
		mIsValid = true;
		setPose(mPose);
	} else if(mSelectionMode == Orientation) {
		mSelectionMode = OrientationVariance;
		mCovPhiXItem->setVisible(true);
		mCovPhiYItem->setVisible(true);
	}

	e->accept();
}

void PoseTool2D::onMouseReleased(QMouseEvent* e) {
	if(e->button() != Qt::LeftButton)
		return;

	if(mSelectionMode == PositionVariance) {
		mSelectionMode = Orientation;
	} else if(mSelectionMode == OrientationVariance) {
		mSelectionMode = Position;
		onNewPose(mPose);
	}

	e->accept();
}

void PoseTool2D::onMouseMoved(QMouseEvent* e) {
	if(mSelectionMode == Position)
		return;

	const QPointF diff = QPointF(getSite()->getViewManager()->mapToScene(e->pos()) - mLastPos);
	float r = hypotf(diff.x(), diff.y());

	if(mSelectionMode == PositionVariance) {
		r = std::max(mMinPositionVariance, r);
		mPose.cov(0,0) = r;
		mPose.cov(1,1) = r;
	} else if(mSelectionMode == Orientation) {
		// Invert the y difference as our y coordinate axis is flipped
		float phi = std::atan2(-diff.y(), diff.x());
		mPose.phi() = -phi;
	} else if(mSelectionMode == OrientationVariance) {
		// Clamp to interval between minvar and 2*PI. Noone would ever need more variance.
		r = std::max(mMinOrientationVariance, std::min(r, mira::two_pi<float>()));
		mPose.cov(2,2) = r;
	}

	setPose(mPose);
	e->accept();
}

void PoseTool2D::activate() {
	VisualizationTool2D::activate();
	mItem->setVisible(false); // Will be activated by left click
}

void PoseTool2D::deactivate() {
	VisualizationTool2D::deactivate();
	mCovPhiXItem->setVisible(false);
	mCovPhiYItem->setVisible(false);
}

///////////////////////////////////////////////////////////////////////////////

void PoseTool2D::init(IVisualizationSite* site) {
	VisualizationTool2D::init(site);

	const float baseZ = getSite()->getToolBaseZValue();
	mItem->setZValue(baseZ + 0.0f);
	mCovXYItem->setZValue(baseZ + 0.0f);
	mCovPhiXItem->setZValue(baseZ + 0.0f);
	mLineXItem->setZValue(baseZ + 0.1f);
	mLineYItem->setZValue(baseZ + 0.1f);
}

QGraphicsItem* PoseTool2D::setupScene(QGraphicsScene* mgr) {
	QPen xpen;
	xpen.setColor(Qt::red);
	xpen.setCosmetic(true);

	QPen ypen;
	ypen.setColor(Qt::green);
	ypen.setCosmetic(true);

	mLineXItem = new QGraphicsLineItem();
	mLineXItem->setPen(xpen);
	mLineYItem = new QGraphicsLineItem();
	mLineYItem->setPen(ypen);
	mCovXYItem = new QGraphicsEllipseItem();
	mCovXYItem->setPen(Qt::NoPen);
	mCovXYItem->setBrush(QColor(255, 255, 255, 128));
	mCovPhiXItem = new QGraphicsPolygonItem();
	mCovPhiXItem->setPen(Qt::NoPen);
	mCovPhiXItem->setBrush(QColor(128, 0, 0, 128));
	mCovPhiXItem->setVisible(false);
	mCovPhiYItem = new QGraphicsPolygonItem();
	mCovPhiYItem->setPen(Qt::NoPen);
	mCovPhiYItem->setBrush(QColor(0, 128, 0, 128));
	mCovPhiYItem->setVisible(false);

	mItem = new QGraphicsItemGroup();
	mItem->addToGroup(mLineXItem);
	mItem->addToGroup(mLineYItem);
	mItem->addToGroup(mCovXYItem);
	mItem->addToGroup(mCovPhiXItem);
	mItem->addToGroup(mCovPhiYItem);
	mItem->setVisible(false);
	mgr->addItem(mItem);

	mSelectionMode = Position;

	// set initial covariances
	mPose.cov.setZero(3,3);
	mPose.cov(0,0) = mMinPositionVariance;
	mPose.cov(1,1) = mMinPositionVariance;
	mPose.cov(2,2) = mMinOrientationVariance;
	setPose(mPose);
	return mItem;
}

void PoseTool2D::setPose(PoseCov2 pose) {
	const float ax = pose.phi();
	const float ay = pose.phi() + pi<float>() * 0.5f;
	const float r = 1.0f;

	// Calculate our offset points with the length
	const QPointF o(pose.x(), pose.y());
	const QPointF x(o.x() + r * cosf(ax), o.y() + r * sinf(ax));
	const QPointF y(o.x() + r * cosf(ay), o.y() + r * sinf(ay));

	mLineXItem->setLine(QLineF(o, x));
	mLineYItem->setLine(QLineF(o, y));

	// Construct 2d matrix, we don't care about the phi variance right now
	Eigen::Matrix3d cov3d = pose.cov.cast<double>();
	Eigen::Matrix2d cov = cov3d.block<2,2>(0, 0);

	// Sanity check against infinitesimally small covariance matrices
	if(isApprox<double>(cov.trace(), 0.0, std::numeric_limits<double>::epsilon()))
		return;

	// See http://www.visiondummy.com/2014/04/draw-error-ellipse-representing-covariance-matrix/ for the math
	Eigen::SelfAdjointEigenSolver<Eigen::Matrix2d> eigensolver(cov);
	if(eigensolver.info() != Eigen::Success)
		return;

	const Eigen::Vector2d values = eigensolver.eigenvalues();
	const Eigen::Matrix2d vectors = eigensolver.eigenvectors();

	// Calculate the size of the minor and major axes and the angle of the ellipsis
	const double chi = 1.0f; // In line with the 3D cov visualization. I would rather use 2.4477 for a 95% confidence interval, hmm...
	const double angle = atan2(static_cast<double>(vectors(1,1)), static_cast<double>(vectors(0,1)));
	const double rx = chi * sqrt(static_cast<double>(values(1))); // We take the bigger one as the major axis
	const double ry = chi * sqrt(static_cast<double>(values(0)));

	// Finally set the ellipse
	mCovXYItem->setPos(static_cast<double>(pose.x()), static_cast<double>(pose.y()));
	mCovXYItem->setRect(-rx, -ry, 2.0 * rx, 2.0 * ry);
	mCovXYItem->setRotation(Radiand(pose.phi() + angle).deg());
	mCovXYItem->setVisible(true);

	// Set visualization of phi variance
	setPhiVariance(pose);
}

void PoseTool2D::setPhiVariance(const PoseCov2& pose) {
	const double var = pose.cov.cast<double>()(2,2);
	const QLineF x = mLineXItem->line();
	const QLineF y = mLineYItem->line();

	mCovPhiXItem->setPolygon(getPhiVariancePolygon(x, var));
	mCovPhiYItem->setPolygon(getPhiVariancePolygon(y, var));
}

QPolygonF PoseTool2D::getPhiVariancePolygon(const QLineF& line, double var) {
	// Get normalized normal vectors
	const qreal norm = qSqrt(line.dx() * line.dx() + line.dy() * line.dy());
	const QPointF n1(norm * -line.dy(), norm * line.dx());
	const QPointF n2(norm * line.dy(), norm * -line.dx());
	const double len = std::pow(10.0, qSqrt(var));

	// Construct our endpoints
	const QPointF p1(line.p2().x() + len * n1.x(), line.p2().y() + len * n1.y());
	const QPointF p2(line.p2().x() + len * n2.x(), line.p2().y() + len * n2.y());

	// Now construct the polygon
	QPolygonF poly;
	poly.append(line.p1());
	poly.append(p1);
	poly.append(p2);
	poly.append(line.p1());
	return poly;
}

///////////////////////////////////////////////////////////////////////////////

}
