/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PointVectorObject.C
 *    Implementation of PointVectorObject.h.
 *
 * @author Christian Reuther
 * @date   2014/09/09
 */

#include <visualization/2d/PointVectorObject.h>

#include <QStyleOptionGraphicsItem>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <typename PointType>
PointVectorObject<PointType>::PointVectorObject(QGraphicsItem* parent) : QAbstractGraphicsShapeItem(parent) {
	setColor(Qt::red);
	setWidth(0.01);
}

template <typename PointType>
PointVectorObject<PointType>::~PointVectorObject() {
	mData.clear();
}

template <typename PointType>
void PointVectorObject<PointType>::setData(const std::vector<PointType>& data) {
	mData.clear();
	mData.reserve(data.size());

	mBoundingRect = QRectF();
	foreach(auto p, data)
		addPoint(static_cast<qreal>(p(0)), static_cast<qreal>(p(1)));
	mBoundingRect = mBoundingRect.normalized();
	setWidth(mWidth);
}

template <typename PointType>
void PointVectorObject<PointType>::setData(const std::vector<std::pair<PointType, QColor>>& data) {
	mData.clear();
	mData.reserve(data.size());

	mBoundingRect = QRectF();
	foreach(auto p, data)
		addPoint(static_cast<qreal>(p.first(0)), static_cast<qreal>(p.first(1)), p.second);
	mBoundingRect = mBoundingRect.normalized();
	setWidth(mWidth);
}

template <typename PointType>
void PointVectorObject<PointType>::setWidth(qreal width) {
	mWidth = qMax<qreal>(0.005, width);
	mBoundingRectAdjusted = mBoundingRect.adjusted(-mWidth, -mWidth, 2 * mWidth, 2 * mWidth);
	updateColor();
}

template <typename PointType>
void PointVectorObject<PointType>::setColor(const QColor& color) {
	mColor = color;
	updateColor();
}

template <typename PointType>
qreal PointVectorObject<PointType>::width() const {
	return mWidth;
}

template <typename PointType>
QColor PointVectorObject<PointType>::color() const {
	return mColor;
}

///////////////////////////////////////////////////////////////////////////////

template <typename PointType>
void PointVectorObject<PointType>::paint(QPainter* painter, const QStyleOptionGraphicsItem* option, QWidget* widget) {
	painter->setPen(pen());
	painter->setBrush(brush());

	foreach(const auto& p, mData) {
		if(p.second.isValid())
			painter->setBrush(QBrush(p.second));

		painter->drawEllipse(p.first, mWidth, mWidth);
	}
}

template <typename PointType>
QRectF PointVectorObject<PointType>::boundingRect() const {
	return mBoundingRectAdjusted; // Adjust for size of our points
}

///////////////////////////////////////////////////////////////////////////////

template <typename PointType>
void PointVectorObject<PointType>::addPoint(qreal x, qreal y, QColor color) {
	mData.push_back(ColouredPoint(QPointF(x, y), color));

	mBoundingRect.setLeft(qMin<qreal>(mBoundingRect.left(), x));
	mBoundingRect.setRight(qMax<qreal>(mBoundingRect.right(), x));
	mBoundingRect.setTop(qMin<qreal>(mBoundingRect.top(), y));
	mBoundingRect.setBottom(qMax<qreal>(mBoundingRect.bottom(),y));
}

template <typename PointType>
void PointVectorObject<PointType>::updateColor() {
	setPen(Qt::NoPen);
	setBrush(QBrush(mColor));
}

///////////////////////////////////////////////////////////////////////////////

template class PointVectorObject<Point2f>;
template class PointVectorObject<Point3f>;

///////////////////////////////////////////////////////////////////////////////

}
