/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ViewPart.C
 *    Implementation of ViewPart.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/01
 */

#include <rcp/ViewPart.h>

#include <QStylePainter>
#include <QStyleOption>
#include <QCloseEvent>

#include <rcp/ViewPartTitleBar.h>

#include <serialization/Serialization.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ViewPart::ViewPart() : mTitleBar(NULL), mSite(NULL)
{
	mTitleBar = new ViewPartTitleBar(this);
	setTitleBarWidget(mTitleBar);

	setAttribute(Qt::WA_DeleteOnClose, true);
	setFocusPolicy(Qt::StrongFocus);
	mIsVisible = true;

	mArea = Qt::NoDockWidgetArea;

	connect(this, SIGNAL(visibilityChanged(bool)),
	        this, SLOT(onVisibilityChanged(bool)));

	connect(this, SIGNAL(dockLocationChanged(Qt::DockWidgetArea)),
	        this, SLOT(onDockLocationChanged(Qt::DockWidgetArea)));

	connect(this, SIGNAL(topLevelChanged(bool)),
			this, SLOT(onTopLevelChanged(bool)));

	connect(mTitleBar->mMinimize, SIGNAL(clicked()),
	        this, SLOT(onMinimizeRequested()));

}

ViewPart::~ViewPart()
{
	emit viewDestroyed(this);
}

void ViewPart::init(IWorkbenchPartSite* site)
{
	mSite = site;
	assert(mSite!=NULL);

	QWidget* ctrl = createPartControl();

	this->setWidget(ctrl);
	ctrl->show();
}

IWorkbenchPartSite* ViewPart::getSite()
{
	return mSite;
}

void ViewPart::activate()
{
	WorkbenchPart::activate();
	mTitleBar->setHighlight(true);
	mIsVisible = true;
}

void ViewPart::deactivate()
{
	WorkbenchPart::deactivate();
	mTitleBar->setHighlight(false);
}

Qt::DockWidgetArea ViewPart::getDefaultDockArea() const
{
	return Qt::LeftDockWidgetArea;
}

bool ViewPart::isVisible() const
{
	return QDockWidget::isVisible() && mIsVisible;
}

Qt::DockWidgetArea ViewPart::getDockArea() const
{
	if(this->isFloating())
		return Qt::NoDockWidgetArea;
	else
		return mArea;
}

QMenu* ViewPart::getViewMenu()
{
	assert(getTitleBar()!=NULL &&
	       getTitleBar()->getMenu()!=NULL);

	return getTitleBar()->getMenu();
}

ViewPartTitleBar* ViewPart::getTitleBar()
{
	assert(mTitleBar!=NULL);
	return mTitleBar;
}

void ViewPart::focusInEvent(QFocusEvent* e)
{
	emit viewActivated(this);
	QDockWidget::focusInEvent(e);
}

void ViewPart::closeEvent(QCloseEvent* event)
{
	QDockWidget::closeEvent(event);

	if (event->isAccepted())
		emit viewClosed(this);
}

void ViewPart::paintEvent(QPaintEvent *event)
{
	// show border if floating
	if (isFloating()) {
		QStylePainter p(this);
		QStyleOptionFrame framOpt;
		framOpt.init(this);
		p.drawPrimitive(QStyle::PE_FrameDockWidget, framOpt);
	} else {
		QPainter p(this);
		QPalette pal = palette();
		QColor baseColor;
		baseColor = pal.color(QPalette::Active, QPalette::Mid);

		p.setPen(QPen(baseColor));
		p.drawRect(0,0,width()-1,height()-1);
	}

}

void ViewPart::onVisibilityChanged(bool visible)
{
	//std::cout << "onVisibilityChanged " << this->windowTitle().toStdString() << " : " << visible << std::endl;
	mIsVisible = visible;
	if(visible)
		emit viewActivated(this);
}

void ViewPart::onMinimizeRequested()
{
	if(!isFloating())
		emit minimizedRequested(this);
}

void ViewPart::onDockLocationChanged(Qt::DockWidgetArea area)
{
	//std::cout << "onDockLocationChanged: " << this << ", "  << mArea << std::endl;
	mArea = area;
	if(!isFloating())
		emit currentDockLocationChanged(mArea);
}

void ViewPart::onTopLevelChanged(bool floating)
{   // called when floating / docked
	//std::cout << "onTopLevelChanged: " << this << ", " << floating << std::endl;

	if(floating) {
		mTitleBar->mMinimize->hide();
		emit currentDockLocationChanged(Qt::NoDockWidgetArea);
	} else {
		mTitleBar->mMinimize->show();
		// no currentDockLocationChanged here, since this will be triggered
		// by onDockLocationChanged(), Qt is pretty inconsitent here
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION( mira::ViewPart, mira::WorkbenchPart );
