/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TreeViewFilter.h
 *    Filter widget containing an line edit that filters the elements
 *    of an assigned TreeView.
 *
 * @author Erik Einhorn
 * @date   2012/02/19
 */

#ifndef _MIRA_TREEVIEWFILTER_H_
#define _MIRA_TREEVIEWFILTER_H_

#include <QWidget>

#include <widgets/GuiWidgetsExports.h>

class QModelIndex;
class QTreeView;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class LineEditClear;

class MIRA_GUI_WIDGETS_EXPORT TreeViewFilter : public QWidget
{
	Q_OBJECT
public:

	TreeViewFilter(QTreeView* view, QWidget* parent=NULL);

protected:

	virtual void timerEvent(QTimerEvent*);

public slots:

	void setAutoHide(bool hide);
	void showFilter();
	void hideFilter();

	/**
	 * Tell this filter to use the special FilterRole of the tree view items
	 * to be used for checking if the filter matches. By default the text
	 * of the DisplayRole is used (which e.g. can be set via item->setText(...)).
	 * However, sometimes it is desirable to do the filtering based on a
	 * different text or for a subset of the items only. In this case, this
	 * method can be called to enable a different mode, which only checks
	 * those items, where the special FilterRole is specified and where
	 * the text of this FilterRole matches the filter.
	 * To specify a FilterRole text, e.g. use
	 * \code
	 * item->setData(column, (Qt::ItemDataRole)TreeViewFilter::FilterRole, text);
	 * \endcode
	 * The FilterRole value is specified by this class.
	 */
	void useFilterRole(bool use=true);

public:
	enum {
		FilterRole = Qt::UserRole+0x1F
	};

public:

	bool getAutoHide() const;

	/**
	 * If set to true, the filter will stay active as long as its filter string is not empty,
	 * i.e. it will filter frequently at a fixed interval, thus also applying to updated content.
	 * (default: false)
	 */
	void setAlwaysActive(bool always);

	/**
	 * If set to true, all (visible) nodes in the tree will be always expanded.
	 * Otherwise, only matching nodes from an active filter will be expanded.
	 * (default: false)
	 */
	void setExpandAll(bool expandAll);

	/**
	 * If set to true, the children of items matching the
	 * entered filter will be visible, otherwise hidden.
	 * (default: true).
	 */
	void setShowChildren(bool show);

	/**
	 * If set to true, the children of items matching the
	 * entered filter (if visible) will be expanded automatically.
	 * (default: true).
	 */
	void setExpandChildren(bool expand);

	/**
	 * Specifies the columns that are used for testing if the pattern matches.
	 * (default: all columns)
	 *
	 * If an empty vector is passed, all columns are used for filtering.
	 */
	void setFilterColumns(const std::vector<int>& cols);
	void setFilterColumns(int col1);
	void setFilterColumns(int col1, int col2);
	void setFilterColumns(int col1, int col2, int col3);

	/**
	 * Clear the filter text.
	 */
	void clearFilter();

private slots:

	void onFilterTextChanged(const QString& text);

private:

	void autoHide();
	bool applyFilter(const QString& filter, const QModelIndex& parent,
	                 bool forceVisible);
	void startFilterUpdateTimer(int delay);

private:

	QTreeView* mTreeView;
	LineEditClear* mFilterEdit;
	int mFilterTimerId;
	int mHideTimerId;

	std::vector<int> mFilterColumns;

	bool mAutoHide;
	bool mAlwaysActive;
	bool mExpandAll;
	bool mShowChildren;
	bool mExpandChildren;
	bool mUseFilterRole;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
