/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization3D.h
 *    Declaration of Visualization3D and related classes.
 *
 * @author Erik Einhorn
 * @date   2010/12/19
 */

#ifndef _MIRA_VISUALIZATION3D_H_
#define _MIRA_VISUALIZATION3D_H_

#include <math/Angle.h>
#include <geometry/Point.h>
#include <visualization/Visualization.h>

//////////////////////////////////////////////////////////////////////////////

// Ogre forward decls
namespace Ogre {
	class SceneManager;
	class Camera;
	class SceneNode;
}

class QMouseEvent;

//////////////////////////////////////////////////////////////////////////////

namespace mira {

class InteractionListener3D;

//////////////////////////////////////////////////////////////////////////////

/**
 * The interface of a Visualization3DSite.
 * A Visualization3DSite is the host or rather the container for visualizations.
 * The IVisualization3DSite interface is e.g. implemented by the
 * Visualization3DView. The Visualization3DSite usually has an internal camera
 * that can be moved freely by the user.
 * The position of that internal camera can be obtained and set via getCamera()
 * and setCamera(). Additionally, the camera can be acquired by a visualization
 * or a tool. The visualization/tool is then responsible for updating the
 * cameras position.
 */
class IVisualization3DSite : public IVisualizationSiteTransformable
{
public:
	virtual ~IVisualization3DSite() {}

	virtual Ogre::SceneManager* getSceneManager() = 0;

	virtual void registerInteractionListener(InteractionListener3D* listener) = 0;
	virtual void removeInteractionListener(InteractionListener3D* listener) = 0;
	virtual std::set<InteractionListener3D*> getInteractionListeners() = 0;

public:

	class MIRA_GUI_VISUALIZATION_EXPORT Camera
	{
	public:

		Camera();

		template <typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("X", position.x(), "the position of the camera");
			r.property("Y", position.y(), "the position of the camera");
			r.property("Z", position.z(), "the position of the camera");
			r.property("Yaw",
			           rad2degGetter(yaw),
			           deg2radSetter(yaw),
			           "the orientation of the camera");
			r.property("Pitch",
			           rad2degGetter(pitch),
			           deg2radSetter(pitch),
			           "the orientation of the camera");
			r.property("Roll",
			           rad2degGetter(roll),
			           deg2radSetter(roll),
			           "the orientation of the camera");
			r.property("Distance", distance, "the zoom or distance");
		}

		void lookAt(const Point3f& from, const Point3f& to);

		// The camera rotation and translation, specified by mouse interaction

		Point3f position;
		float yaw, pitch, roll;
		float distance;
	};

	/**
	 * Obtains the current camera settings (position, orientation, etc.) of the
	 * views own camera.
	 */
	virtual const Camera& getCamera() const = 0;

	/**
	 * Sets the current camera settings (position, orientation, etc.) of the
	 * views own camera.
	 */
	virtual void setCamera(const Camera& camera) = 0;

	/**
	 * Requests the camera of the visualization site.
	 * This method can be called by visualizations or tools to get direct
	 * access to the camera. The visualization/tool is then responsible for
	 * updating the cameras position (e.g. in its update() method).
	 * Optionally, a callback can be specified, that is called if another
	 * visualization/tool has requested the camera.
	 * The visualization finally should release the camera by calling
	 * releaseCamera().
	 */
	virtual Ogre::Camera* acquireCamera(boost::function<void()> lostCameraCallback = boost::function<void()>()) = 0;

	/**
	 * Releases a camera that previously was acquired.
	 */
	virtual void releaseCamera() = 0;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all 3D visualization that are used by
 * Visualization3DView.
 */
class MIRA_GUI_VISUALIZATION_EXPORT Visualization3D :
	public Visualization
{
	MIRA_ABSTRACT_OBJECT(Visualization3D)
public:

	Visualization3D();
	virtual ~Visualization3D() {}

public:

	/**
	 * Is called by the visualization view when the visualization is attached.
	 * The default implementation stores the pointer to the IVisualization3DSite
	 * interface that can be retrieved using the getSite() method for later use.
	 * Moreover, it calls the setupScene() method.
	 * You should never need to overwrite this method in derived classes. Use
	 * the setupScene() method instead. However, if you overwrite this method
	 * you SHOULD call this default implementation.
	 *
	 * The passed interface must be of the type IVisualization3DSite.
	 */
	virtual void init(IVisualizationSite* site);

public:

	/**
	 * Returns a pointer to the IVisualization3DSite interface that was set
	 * in the init() method.
	 * Or returns NULL, if init() was not yet called.
	 */
	virtual IVisualization3DSite* getSite();

public:

	/**
	 * Must be implemented in derived classes. Is called within the
	 * visualization's init() method. You can use this method to create
	 * your scene and scene objects.
	 * The passed IVisualization3DSite interface is the same that is
	 * returned by getSite().
	 */
	virtual void setupScene(IVisualization3DSite* site) = 0;

	/**
	 * Must be implemented in derived classes to return the scene node
	 * to which this visualization attaches its objects.
	 */
	virtual Ogre::SceneNode* getNode() = 0;

private:

	IVisualization3DSite* mSite;

};

//////////////////////////////////////////////////////////////////////////////

}

#endif
