/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization.h
 *    Declaration of Visualization and related classes.
 *
 * @author Erik Einhorn
 * @date   2010/12/30
 */

#ifndef _MIRA_VISUALIZATION_H_
#define _MIRA_VISUALIZATION_H_

#include <factory/Factory.h>
#include <utils/Time.h>

#include <visualization/ChannelProperty.h>

#include <visualization/GuiVisualizationExports.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * The primary interface between a Visualization and the container of the
 * visualization (e.g. the VisualisationView)
 */
class IVisualizationSite// : public IAuthorityProvider
{
public:
	virtual ~IVisualizationSite() {}
};

/**
 * The interface between a Visualization and the container of the
 * visualization (e.g. the VisualisationViewTransformable) that supports a target frame
 */
class IVisualizationSiteTransformable : public IVisualizationSite
{
public:
	virtual ~IVisualizationSiteTransformable() {}
	
	virtual const std::string& getFixedFrame() const = 0;
	virtual const std::string& getCameraFrame() const = 0;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all derived visualizations, namely Visualization3D
 * and Visualization2D.
 */
class MIRA_GUI_VISUALIZATION_EXPORT Visualization :
	public Object, public IAuthorityProvider, public Authority
{
	MIRA_ABSTRACT_OBJECT(Visualization)
public:

	Visualization();
	virtual ~Visualization() {};

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.roproperty("Status", &getStatusManager(), "The status of this visualization");
		r.member("Enabled", mEnabled, "Is true if the visualization is enabled and visible");
		r.member("Name", mDisplayedName, "The name that is shown to the user");
	}

public:

	/**
	 * Is called by the visualization view when the visualization is attached.
	 * The Visualization should initialize itself. The default implementation
	 * stores the pointer to the IVisualizationSite interface that can be
	 * retrieved using the getSite() method for later use.
	 * If you overwrite this method you SHOULD call the implementation of the
	 * base class.
	 */
	virtual void init(IVisualizationSite* site) = 0;

	/**
	 * Returns a pointer to the IVisualizationSite interface that was set
	 * in the init() method.
	 * Or returns NULL, if init() was not yet called.
	 */
	virtual IVisualizationSite* getSite() = 0;

public:

	/**
	 * Is called regularly by the VisualizationView to update the visualization.
	 * The time span since the last call of update is passed as parameter.
	 * It calls update() that can be overwritten in derived classes
	 */
	virtual void processUpdate(Duration dt);

public:

	/// Returns true, if this visualization is enabled and visible
	bool isEnabled() const { return mEnabled; }

	/**
	 * Changes the enabled-state and therefore the visibility of this
	 * visualization.
	 * Derived classes should overwrite this method to hide and show
	 * the visualization as desired. When overwriting this method you
	 * should always call the method of the base class.
	 */
	virtual void setEnabled(bool enabled) { mEnabled = enabled; }

public:

	/// Returns the associated name of this visualization.
	const std::string& getName() const { return mDisplayedName; }

	/// Sets the name of this visualization.
	void setName(const std::string& name) { mDisplayedName = name; }

protected:

	/**
	 * This method can be implemented in the derived Visualization classes.
	 * See processUpdate()
	 */
	virtual void update(Duration dt) {
		onUpdate(dt); // for backward compatibility.
	};

	/**
	 * Provided for backward compatibility.
	 * \deprecated Please overwrite the update() method instead!
	 */
	virtual void onUpdate(Duration dt) {};

public:

	/**
	 * The data structure that is returned by getDataConnection. It provides
	 * information about the direct data connection for this visualization,
	 * i.e. the type of the data which this visualization can show and the
	 * main property channel that needs to be set to the data channel.
	 */
	struct DataConnection
	{

		DataConnection() : property(NULL) {}

		template<typename T>
		DataConnection(ChannelProperty<T>& iProperty) :
			type(typeName<T>()), property(&iProperty) {}

		Typename type;
		ChannelPropertyBase* property;
	};

	/**
	 * Returns information about the direct data connection for this
	 * visualization. This information is used to support drag'n drop of
	 * channels into a visualization window to find the correct visualization
	 * for that data. Moreover, it is used in the tape player preview, etc.
	 * If your visualization should support these features you must implement
	 * this method by returning a valid DataConnection structure. By default
	 * an empty DataConnection structure is returned.
	 */
	virtual DataConnection getDataConnection() { return DataConnection(); }


	/**
	 * Sets up the data connection channel property (if any) with the specified
	 * channel id. The property is added automatically to the channel property
	 * update set. Which is used for regular update of the properties.
	 * This method is for internal use, and called by VisualizationView upon
	 * a drag'n'drop action.
	 */
	void setupDataConnectionChannel(const std::string& channelID)
	{
		DataConnection connection = getDataConnection();
		if(connection.property==NULL) // if no data connection channel, abort here
			return;

		// set the id
		connection.property->set(channelID, this);
		// and add the property to our property set, for regularl update of the properties
		mChannelProperties.insert(connection.property);
	}

protected:

	enum OptionalMode
	{
		NOT_REQUIRED,
		REQUIRED
	};

	/**
	 * Special property-method that should be called within the reflect method
	 * to specify a ChannelProperty.
	 *
	 * E.g.:
	 * @code
	 * template <typename Reflector>
	 * void reflect(Reflector& r)
	 * {
	 *     channelProperty(r, "MyChannel", myChannelProperty, "This is a comment to my channel");
	 * }
	 * @endcode
	 */
	template<typename T, typename Reflector>
	void channelProperty(Reflector& r, const std::string& name,
			ChannelProperty<T>& channel, const std::string& comment, OptionalMode optional=REQUIRED)
	{
		ChannelProperty<T>::channelProperty(r,name, channel,comment,this);
		mChannelProperties.insert(&channel);
		channel.setName(name);
		if(optional==NOT_REQUIRED)
			channel.setOptional();
		r.roproperty(std::string(name+" Updates").c_str(),
		             getter<uint32>(boost::bind(&ChannelProperty<T>::getDataUpdateCount, &channel)),
		             "How many data updates were received");
	}

private:
	std::set<ChannelPropertyBase*> mChannelProperties;
	bool mEnabled;
	std::string mDisplayedName; // the name of this visualization that is shown to the user

public: // implementation of IAuthorityProvider

	virtual Authority& getAuthority();

};

//////////////////////////////////////////////////////////////////////////////

}

#endif
