/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseTool2D.h
 *    Tool for specifying a PoseCov2 in the 2D graphics view.
 *
 * @author Christian Reuther
 * @date   2018/08/29
 */

#ifndef _MIRA_POSETOOL2D_H_
#define _MIRA_POSETOOL2D_H_

#include <visualization/VisualizationTool2D.h>

#include <transform/Pose.h>

#include <QGraphicsPathItem>
#include <QGraphicsLineItem>
#include <QGraphicsEllipseItem>
#include <QGraphicsItemGroup>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class MIRA_GUI_VISUALIZATION_EXPORT PoseTool2D : public VisualizationTool2D
{
	enum SelectionMode {
		Position,
		PositionVariance,
		Orientation,
		OrientationVariance,
	};

public:
	PoseTool2D(float minPositionVariance = 0.1f, float minOrientationVariance = 0.25f);
	virtual ~PoseTool2D();

	template<typename Reflector> void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, VisualizationTool2D);
	}

public:
	virtual void onMousePressed(QMouseEvent* e);
	virtual void onMouseReleased(QMouseEvent* e);
	virtual void onMouseMoved(QMouseEvent* e);

	virtual void activate();
	virtual void deactivate();

protected:
	virtual void init(IVisualizationSite* site);
	virtual QGraphicsItem* setupScene(QGraphicsScene* mgr);

protected:
	/**
	 * @brief A new pose was selected by the use.
	 *
	 * This method should be overwritten to be notified whenever the user has
	 * selected a new pose
	 *
	 * @param[in] pose The new pose selected by the user.
	 */
	virtual void onNewPose(const PoseCov2& pose) {}

protected:
	void setPose(PoseCov2 pose);
	void setPhiVariance(const PoseCov2& pose);
	QPolygonF getPhiVariancePolygon(const QLineF& line, double var);

	QGraphicsItemGroup* mItem;
	QGraphicsLineItem* mLineXItem;
	QGraphicsLineItem* mLineYItem;
	QGraphicsEllipseItem* mCovXYItem;
	QGraphicsPolygonItem* mCovPhiXItem;
	QGraphicsPolygonItem* mCovPhiYItem;

	SelectionMode mSelectionMode;
	QPointF mLastPos;
	bool mIsValid;

	float mMinPositionVariance;
	float mMinOrientationVariance;

	PoseCov2 mPose;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
