/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DynamicPropertiesTest.C
 *    Test dynamic properties
 *
 * @author Christof Schröter
 * @date   2020/06/24
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>
#include <fw/UnitManager.h>

#include <serialization/PropertyNode.h>
#include <fw/RemoteAuthority.h>

using namespace mira;

// Testing dynamic properties: the dynamic properties units change their properties
// in a defined pattern.
// Tester classes subscribe to the PropertyNodes and check that they always see
// a valid properties state, and that all expected states are seen.

/**
 * TestNode is a representation of PropertyNode structure and IDs.
 * It is initialized with a PropertyNode, then updated with
 * PropertyNodeListener messages. By checking the final state,
 * correctness and completeness of the messages is verified.
 */
struct TestNode
{
	typedef boost::shared_ptr<TestNode> TestNodePtr;

	TestNode(PropertyNode* iNode, bool recursive = false) : node(iNode), id(iNode->id())
	{
		if (recursive) {
			foreach(auto c, node->children())
				children.emplace_back(new TestNode(c, true));
		}
	}

	bool addChildren(const PropertyNode* node, int index, int count) {
		if (node == this->node) {
//			std::cout << "-------------------- addChildren " << node->id() << " " << index << " " << count << std::endl;
			PropertyNode::NodeList::const_iterator start = std::next(node->children().begin(), index);
			PropertyNode::NodeList::const_iterator end = std::next(start, count);

			std::vector<TestNodePtr> newNodes;
			for (auto it = start; it != end; ++it)
				newNodes.emplace_back(new TestNode(*it));

			children.insert(std::next(children.begin(), index), newNodes.begin(), newNodes.end());
			return true;
		}

		foreach(auto c, children) {
			if (c->addChildren(node, index, count))
				return true;
		}

		return false;
	}	

	bool removeChildren(const PropertyNode* node, int index, int count) {
		if (node == this->node) {
//			std::cout << "-------------------- removeChildren " << node->id() << " " << index << " " << count << std::endl;
			std::vector<TestNodePtr>::iterator start = std::next(children.begin(), index);
			std::vector<TestNodePtr>::iterator end = std::next(start, count);
			children.erase(start, end);
			return true;
		}

		foreach(auto c, children) {
			if (c->removeChildren(node, index, count))
				return true;
		}

		return false;
	}	

	bool moveChildren(const PropertyNode* node, int index, int count, int destination) {
		if (node == this->node) {
//			std::cout << "-------------------- moveChildren" << node->id() << " " << index << " " << count << " " << destination << std::endl; << std::endl;
			std::vector<TestNodePtr>::iterator first, newFirst, end;

			if (index < destination) {
				first = std::next(children.begin(), index);
				newFirst = std::next(first, count);
				end = std::next(children.begin(), destination);
			} else {
				first = std::next(children.begin(), destination);
				newFirst = std::next(children.begin(), index);
				end = std::next(newFirst, count);
			}

			std::rotate(first, newFirst, end);
			return true;
		}

		foreach(auto c, children) {
			if (c->moveChildren(node, index, count, destination))
				return true;
		}

		return false;
	}	

	void display(int indent) {
		for (int i = 0; i < indent; ++i)
			std::cout << " ";
		std::cout << id << std::endl;
		foreach(auto c, children)
			c->display(indent+2);
	}

	PropertyNode* node;
	std::string id;

	std::vector<TestNodePtr> children;
};

struct RemotePropertyNode : public AbstractRemotePropertyNode
{
	RemotePropertyNode(const PropertyNodeInfo& info) :
		AbstractRemotePropertyNode(info) {}

	virtual void setFromJSON(const json::Value& node) {}
	virtual json::Value getAsJSON() const { return json::Value(); }
	virtual void synchronize() {}
};

void compare(boost::shared_ptr<TestNode> ids, PropertyNode* properties)
{
	BOOST_CHECK_EQUAL(ids->id, properties->id());
	BOOST_REQUIRE(ids->children.size() == properties->children().size());

	auto i = ids->children.begin();
	auto p = properties->children().begin();
	for (; i != ids->children.end(); ++i, ++p)
		compare(*i, *p);
}

class DynTester : public PropertyNodeListener
{
public:
	DynTester(SharedAuthority authority) : mAuthority(authority), mProperties((*mAuthority).getProperties()) {
//		std::cout << "Testing authority " << (*mAuthority).getGlobalID() << std::endl;
		MIRA_FW.getAuthorityManager().stop((*mAuthority).getGlobalID());
		mProperties->getRootNode()->registerListener(this);

		mTestIDs.reset(new TestNode(&(*mProperties), true));
		mStatusNode = mProperties->findChildNode("DynStatus")->toTyped<int>();
		BOOST_REQUIRE((mStatusNode != NULL));

		while (true) {
			try {
				mStatus = mStatusNode->get();
				MIRA_SLEEP(1000);
				break;
			}
			catch(XIO& ex) {}
		}

//		mTestIDs->display(2);
	}

	~DynTester() {
		mProperties->getRootNode()->unregisterListener(this);
	}

	bool seenAllStatus() { return mStatusTested.size() == 4; }
	bool seenAllStatusTransitions() { return mStatusTransitionTested.size() == 16; }

	void testStepLocal(int n) {
		std::cout << "step " << n << " -----------------------------------------------------------------------" << std::endl;

		MIRA_FW.getAuthorityManager().start((*mAuthority).getGlobalID());
		MIRA_SLEEP(300);
		// stop to fix the property state for comparison
		MIRA_FW.getAuthorityManager().stop((*mAuthority).getGlobalID());

//		std::cout << "DynStatus: " << mStatusNode->get() << std::endl;
		mTestIDs->display(2);

		compare(mTestIDs, mProperties.get());

		mStatusTested.insert(mStatusNode->get());
	}

	void testStepRemote(int n, int targetStatus) {
		std::cout << "step " << n << " -----------------------------------------------------------------------" << std::endl;
//		std::cout << mStatusTransitionTested.size() << " transitions" << std::endl;
//		std::cout << print(mStatusTransitionTested) << std::endl;

		int status = mStatus;
		while (targetStatus != status) {
			MIRA_FW.getAuthorityManager().start((*mAuthority).getGlobalID());
			MIRA_SLEEP(((targetStatus + 4 - status) % 4) * 500 - 400);
			// stop to synch to this state next
			MIRA_FW.getAuthorityManager().stop((*mAuthority).getGlobalID());

			json::Value v = MIRA_FW.getRPCManager().call<json::Value>((*mAuthority).getGlobalID()+"#builtin",
			                                                           "getPropertyJSON", std::string("DynStatus")).get();
			status = v.get_int();
		}
//		std::cout << "DynStatus: " << status << std::endl;

		mStatusNode->get(); // this will trigger to get the value from the remote authority, and also fetch the structure
		                    // while the "real" remote authority may have done multiple updates between start and stop,
		                    // the RemoteAuthority (= our local proxy) only fetches an update once, now!

		MIRA_SLEEP(1200) // structure will only be fetched (then updated & signalled) every 1s, so need some minimum wait before the next get()

		mTestIDs->display(2);

		// get the properties directly from the remote authority
		PropertyTree tree = MIRA_FW.getRPCManager().call<PropertyTree>((*mAuthority).getGlobalID()+"#builtin",
		                                                                "getProperties").get();
		RemotePropertyNode* root = new RemotePropertyNode(tree.getRootNodeInfo());
		tree.generatePropertyNodes<RemotePropertyNode>(root);

		compare(mTestIDs, root);
		delete root;

		mStatusTransitionTested.insert(std::make_pair(mStatus, status));
		mStatus = status;
	}

private:

	/// PropertyNodeListener implementation
	virtual void beginAddChildren(const PropertyNode* node, int index, int count) {
		mEditingNode = node;
		mEditingIndex = index;
		mEditingCount = count;
	}

	/// PropertyNodeListener implementation
	virtual void endAddChildren() {
		mTestIDs->addChildren(mEditingNode, mEditingIndex, mEditingCount);
	}

	/// PropertyNodeListener implementation
	virtual void beginRemoveChildren(const PropertyNode* node, int index, int count) {
		mTestIDs->removeChildren(node, index, count);
	}

	/// PropertyNodeListener implementation
	virtual void endRemoveChildren() {}

	/// PropertyNodeListener implementation
	virtual void beginMoveChildren(const PropertyNode* node, int index, int count, int destination) {
		mTestIDs->moveChildren(node, index, count, destination);		
	}

	/// PropertyNodeListener implementation
	virtual void endMoveChildren() {}

private:
	SharedAuthority mAuthority;
	boost::shared_ptr<PropertyNode> mProperties;

	boost::shared_ptr<TestNode> mTestIDs;
	TypedPropertyNode<int>* mStatusNode;

	const PropertyNode* mEditingNode;
	int mEditingIndex, mEditingCount, mEditingDestination;

	int mStatus;
	std::set<int> mStatusTested;
	std::set<std::pair<int, int>> mStatusTransitionTested;
};

class ContainersTester : public PropertyNodeListener
{
public:
	ContainersTester(SharedAuthority authority) : mAuthority(authority), mProperties((*mAuthority).getProperties()) {
//		std::cout << "Testing authority " << (*mAuthority).getGlobalID() << std::endl;
		mVector0Node = mProperties->findChildNode("BigVector.item[0]")->toTyped<int>();
		mProperties->getRootNode()->registerListener(this);
	}

	~ContainersTester() {
		mProperties->getRootNode()->unregisterListener(this);
	}

	void testVector() {
		try {
			BOOST_CHECK_EQUAL(mVector0Node->get(), 0);
		}
		catch(XIO& ex) {
			const std::string expect = "Property is not available (yet?)";
			const std::size_t s = expect.size();
			BOOST_CHECK_EQUAL(std::string(ex.what()).substr(0, s), expect);
		}
	}

	void testList() {
		PropertyNode* node;
		for (int n=0; n<5; ++n) {
			try {
				boost::mutex::scoped_lock lock(mMutex);
				node = mProperties->findChildNode("FluidList.item[" + toString(n) + "].n");
				if (node) {
					BOOST_CHECK_EQUAL(json::write(node->getAsJSON()), toString(n));
				}
				node = mProperties->findChildNode("FluidList.item[" + toString(n) + "].n2");
				if (node) {
					BOOST_CHECK_EQUAL(json::write(node->getAsJSON()), toString(n));
				}
			}
			catch(XIO& ex) {
				const std::string expect = "Property is not available (yet?)";
				const std::size_t  s = expect.size();
				BOOST_CHECK_EQUAL(std::string(ex.what()).substr(0, s), expect);
			}
		}
	}

private:

	/// PropertyNodeListener implementation
	virtual void beginAddChildren(const PropertyNode* node, int index, int count) {}
	virtual void endAddChildren() {}
	virtual void beginRemoveChildren(const PropertyNode* node, int index, int count) {
		mMutex.lock();
	}
	virtual void endRemoveChildren() {
		mMutex.unlock();
	}
	virtual void beginMoveChildren(const PropertyNode* node, int index, int count, int destination) {}
	virtual void endMoveChildren() {}

private:
	SharedAuthority mAuthority;
	boost::shared_ptr<PropertyNode> mProperties;

	TypedPropertyNode<int>* mVector0Node;

	boost::mutex mMutex;
};

BOOST_AUTO_TEST_CASE(DynamicPropertiesTest)
{
	const char* argv[] = {"DynamicPropertiesTest",
	                      "${find DynamicPropertiesTest.xml}",
	                      "-v", "CycleTime=500"};
	Framework fw(4,(char**)argv);
	fw.load();
	fw.start();

	{
		SharedAuthority containersUnit = MIRA_FW.getAuthorityManager().getAuthority("/ContainersUnit");
		BOOST_REQUIRE(containersUnit != NULL);

		ContainersTester tester(containersUnit);

		for (int n = 0; n < 50; ++n) {
			tester.testVector();
			tester.testList();
			MIRA_SLEEP(500);
		}
	}
	{
		SharedAuthority localDynUnit = MIRA_FW.getAuthorityManager().getAuthority("/LocalDynUnit");
		DynTester tester(localDynUnit);

		int n=0;
		while (!tester.seenAllStatus()) {
			tester.testStepLocal(n++);
		}
	}
	{
		SharedAuthority remoteDynUnit = MIRA_FW.getAuthorityManager().getAuthority("/RemoteDynUnit");
		BOOST_REQUIRE(remoteDynUnit != NULL);

		DynTester tester(remoteDynUnit);

		// go through all combinations of "from - to"
		std::vector<int> target = {0, 0, 1, 1, 0, 2, 1, 3, 0, 3, 1, 2, 3, 2, 2, 0, 3, 3};
		int n=0;
		while (!tester.seenAllStatusTransitions()) {
			tester.testStepRemote(n, target[n % target.size()]);
			++n;
		}
	}
}
