/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelPromoteByTypename.C
 *    Implementation of ChannelPromoteByTypename.h.
 *
 * @author Christof Schröter
 * @date   2018/08/26
 */

#include <fw/Framework.h>
#include <fw/ChannelPromoteByTypename.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

bool ChannelPromoteByTypename::promoteChannel(ConcreteChannel<void>* channel,
                                              const Typename& channelType)
{
#ifndef MIRA_DISABLE_CHANNELPROMOTERS
	if (mInstantiatedPromoters.count(channelType) > 0) {
		TypedChannelPromoterBasePtr promoter;
		promoter = mInstantiatedPromoters[channelType];
		MIRA_LOG(NOTICE) << "ChannelPromoteByTypename: Use previously instantiated promoter "
		                 << promoter->getClass().getIdentifier()
		                 << " to promote void channel " << channel->getID() << ".";
		try {
			promoter->promote(channel);
			return true;
		} catch(...) {}
		MIRA_LOG(WARNING) << "ChannelPromoteByTypename: " << promoter->getClass().getIdentifier()
		                  << " did not succeed to promote channel " << channel->getID() << "."
		                  << " Will try other promoters.";
	}

	boost::mutex::scoped_lock lock(mMutex);
	lookForNewPromoters();

	auto r = mKnownPromoters.equal_range(channelType);

	for (auto it = r.first; it != r.second; ++it)
	{
		try {
			MIRA_LOG(NOTICE) << "ChannelPromoteByTypename: Create " << it->second.getIdentifier()
			                 << " to promote " << "void channel " << channel->getID() << ".";
			TypedChannelPromoterBasePtr promoter
				= TypedChannelPromoterBasePtr(it->second.newInstance<TypedChannelPromoterBase>());
			promoter->promote(channel);
			mInstantiatedPromoters[channelType] = promoter;
			return true;
		} catch(...) {}
	}

	MIRA_LOG(WARNING) << "ChannelPromoteByTypename: Could not promote channel " << channel->getID()
	                  << " to type " << channelType << ". Channel remains untyped.";

#endif  // immediately return false if MIRA_DISABLE_CHANNELPROMOTERS
	return false;
}

bool ChannelPromoteByTypename::checkForPromoter(const Typename& channelType, const std::string& channelID)
{
#ifndef MIRA_DISABLE_CHANNELPROMOTERS

	{
		boost::mutex::scoped_lock lock(mMutex);
		lookForNewPromoters();
	}

	bool havePromoter = (mKnownPromoters.count(channelType) > 0);

	if (!havePromoter) {
	#ifdef NDEBUG
		MIRA_LOG(WARNING) << "ChannelPromoteByTypename: Type " << channelType << " (on channel "
		                  << channelID << ") is not registered as a promotable channel type.";
	#else
		// we are in a debug build
		try {
			std::string source = getSourceHint();

			if (source.empty()) {
				MIRA_LOG(WARNING) << "ChannelPromoteByTypename: Type " << channelType << " (on channel "
					          << channelID << ") is not registered as a promotable channel type.";
			} else {
				MIRA_LOG(WARNING) << "ChannelPromoteByTypename: Type " << channelType << " (on channel "
				                  << channelID << ") is not registered as a promotable channel type. "
				                  << "Source hint: " << source << ".";
			}
		} catch (...) {
			MIRA_LOG(WARNING) << "ChannelPromoteByTypename: Type " << channelType << " (on channel "
			                  << channelID << ") is not registered as a promotable channel type.";
		}	
	#endif
	}

	return havePromoter;
#else   // immediately return false if MIRA_DISABLE_CHANNELPROMOTERS
	return false;
#endif
}

void ChannelPromoteByTypename::lookForNewPromoters()
{
	auto promoters = TypedChannelPromoterBase::CLASS().getDerivedClasses();
	if (promoters.size() <= mKnownPromoterClasses.size())
		return; // no new classes

	foreach(auto c, promoters)
	{
		if (mKnownPromoterClasses.count(c.second) > 0)
			continue; // we know this one already

		mKnownPromoterClasses.insert(c.second);
		mKnownPromoters.insert(std::make_pair(c.second.getMetaInfo("PromoteTo"), c.second));
	}
}

std::string ChannelPromoteByTypename::getSourceHint()
{
#ifndef MIRA_DISABLE_CHANNELPROMOTERS
#ifndef NDEBUG
	// get the stacktrace and find the call to publish<> as a hint where to add the registration
	CallStack callstack = CallStack::backtrace();

	size_t p;
	for (p = 0; p < callstack.size(); ++p) {
		//std::cout << callstack[n] << std::endl;
		if (strstr(callstack[p].nameDemangled.c_str(), "mira::ChannelManager::publish"))
			break;
	}
	if (p < callstack.size()) { // found the ChannelManager::publish()
		++p;
		while ((p < callstack.size()) &&
		       (strstr(callstack[p].nameDemangled.c_str(), "mira::ChannelManager::publish") ||
		        strstr(callstack[p].nameDemangled.c_str(), "mira::Authority::publish")         ))
			++p;
		if (p < callstack.size()) // found publish call into framework
			return callstack[p].nameDemangled + " at " + callstack[p].sourceLocation();
	}
#endif
#endif  // immediately return "" if MIRA_DISABLE_CHANNELPROMOTERS or NDEBUG
	return std::string();
}

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_REGISTER(mira::TypedChannelPromoterBase, mira::Object)
