/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ConcreteChannel.hpp
 *    Implementation header for all template functions of the
 *    ConcreteChannel class.
 *
 * @author Christof Schröter
 * @date   2019/04/04
 */

#ifndef _MIRA_CONCRETECHANNELIMPL_H_
#define _MIRA_CONCRETECHANNELIMPL_H_

#include <fw/Framework.h>
#include <fw/FrameworkTransformer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <typename T>
ChannelRead<T> ConcreteChannel<T>::read()
{
	ChannelBufferBase::Slot* s = mBuffer->readNewestSlot();
	if(s==NULL)
		MIRA_THROW(XInvalidRead, "Cannot read from channel '"
		          << mID << "' since there is no data");
	return ChannelRead<T>(this, Buffer::castSlot(s));
}

template <typename T>
ChannelRead<T> ConcreteChannel<T>::read(const Time& timestamp, SlotQueryMode mode,
                                        const Duration& tolerance)
{
	ChannelBufferBase::Slot* s = mBuffer->readSlotAtTime(timestamp, mode);
	if(s==NULL)
		MIRA_THROW(XInvalidRead, "Cannot read slot from channel '"
		           << mID << "' at the specified time: '" << timestamp << "'");

	if(mira::abs(s->timestamp()-timestamp) > tolerance)
	{
		// !!DO NOT REMOVE THIS DUMMY!! It is needed to unlock the slot automatically!
		ChannelRead<T> dummy(this, Buffer::castSlot(s));
		MIRA_THROW(XInvalidRead, "Cannot read slot from channel '"
		           << mID << "', the nearest slot at timestamp '"
		           << s->timestamp() << "' is too far away from the specified time: '"
		           << timestamp << "'");
	}
	return ChannelRead<T>(this, Buffer::castSlot(s));
}

template <typename T>
ChannelReadInterval<T> ConcreteChannel<T>::readInterval(const Time& timestamp, std::size_t nrSlots,
                                                        std::size_t olderSlots, std::size_t newerSlots,
                                                        IntervalFillMode fillMode)
{
	std::list<ChannelBufferBase::Slot*> slotList;
	try
	{
		mBuffer->readInterval(timestamp, nrSlots, olderSlots, newerSlots,
		                      fillMode, slotList);
	}
	catch(Exception &ex)
	{
		MIRA_RETHROW(ex, "while trying to read interval ["
		             << olderSlots << " < " << timestamp << " <= " << newerSlots << "] of " << nrSlots
		             << " slots from channel '" << mID << "'")
	}
	return ChannelReadInterval<T>(this, slotList);
}

template <typename T>
ChannelReadInterval<T> ConcreteChannel<T>::readInterval(const Time& from, const Time& to)
{
	std::list<ChannelBufferBase::Slot*> slotList;
	try
	{
		mBuffer->readInterval(from, to, slotList);
	}
	catch(Exception &ex)
	{
		MIRA_RETHROW(ex, "while trying to read interval (" << from << "," << to
		             << "] from channel '" << this->getID() << "'")
	}
	return ChannelReadInterval<T>(this, slotList);
}

template <typename T>
ChannelWrite<T> ConcreteChannel<T>::write()
{
	if ((this->getNrOfSlots() == 0) && !this->isTyped()) {
		if (Framework::instance().getChannelManager().autoPromoteChannelsEnabled())
			ChannelPromoteByTypename::instance().promoteChannel(this, getTypename());
	}

	ChannelBufferBase::Slot* s = mBuffer->requestWriteSlot();
	return ChannelWrite<T>(this, Buffer::castSlot(s));
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
