/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServiceInterfaceHandler.h
 *    Signal handler class for signaling when service interfaces get available.
 *
 * @author Tim Langner
 * @date   2012/03/28
 */

#ifndef _MIRA_SERVICEINTERFACEHANDLER_H_
#define _MIRA_SERVICEINTERFACEHANDLER_H_

#include <queue>
#include <fw/FrameworkExports.h>

#include <fw/DispatcherThread.h>

#include <rpc/AbstractInterfaceCallbackHandler.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * ServiceInterfaceHandler implements a AbstractInterfaceCallbackHandler.
 * It can be added to a DispatcherThread as a signal handler to allow handling
 * of the announcement of new services that implement a special interface in
 * the thread of the dispatcher.
 */
class MIRA_FRAMEWORK_EXPORT ServiceInterfaceHandler :
	public AbstractInterfaceCallbackHandler,
	public DiagnosticRunnable
{
public:

	typedef boost::function<void (const std::string&, const std::string&)> Callback;

	ServiceInterfaceHandler(DispatcherThread* dispatcher, DiagnosticsModulePtr module = NULL) :
		DiagnosticRunnable(module), mDispatcher(dispatcher)  {}

	/// Implemented from AbstractInterfaceCallbackHandler
	virtual void newServiceInterface(const std::string& interface,
	                                 const std::string& service);

	/// Implemented from IRunnable
	virtual void run(DispatcherThread* dispatcher);

	void registerCallback(const std::string& interface, Callback cb);

private:

	DispatcherThread* mDispatcher;
	boost::mutex mMutex;
	std::queue<std::pair<std::string, std::string>> mNewServiceQueue;
	std::multimap<std::string, Callback> mCallbacks;
};

typedef boost::shared_ptr<ServiceInterfaceHandler> ServiceInterfaceHandlerPtr;

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
