/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * All rights reserved.
 *
 * Redistribution and modification of this code is strictly prohibited.
 *
 * IN NO EVENT SHALL "MLAB" BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE
 * OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLABS" HS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN
 * "AS IS" BASIS, AND "MLAB" HAS NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteConnectionPool.h
 *    Connection pool that holds the ownership for RemoteConnections.
 *    Creates move-only ConnectionProxies that will stop their thread in their destructor.
 *
 * @author Tom Mehner
 * @date   Feb 3, 2021
 */

#ifndef _MIRA_FRAMEWORK_INCLUDE_FW_REMOTE_INTERNAL_REMOTECONNECTIONPOOL_H_
#define _MIRA_FRAMEWORK_INCLUDE_FW_REMOTE_INTERNAL_REMOTECONNECTIONPOOL_H_

#include <unordered_map>

#include <serialization/IsTransparentSerializable.h>
#include <fw/RemoteConnection.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// forward declarations
class RemoteOutgoingConnectionBase;
class RemoteIncomingConnection;
class RemoteConnection;
struct KnownFramework;

class RemoteConnectionPool;

/**
 * A proxy object that represents a connection.
 * It is noncopyable.
 */
class RemoteConnectionProxy : boost::noncopyable
{
	friend class RemoteConnectionPool;
public:
	RemoteConnectionProxy() : mRemoteConnectionPool(nullptr), mRemoteConnection(nullptr) {};
	RemoteConnectionProxy (RemoteConnectionProxy&&) noexcept;
	RemoteConnectionProxy& operator= (RemoteConnectionProxy&&) noexcept;

	/**
	 * Destructor
	 * Calls \ref RemoteConnectionPool::stopConnection on its connection
	 */
	~RemoteConnectionProxy();

	friend void swap(RemoteConnectionProxy& a, RemoteConnectionProxy& b);

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.delegate(mRemoteConnection);
	}

	const RemoteConnection* operator->() const { return mRemoteConnection; }
	RemoteConnection* operator->() { return mRemoteConnection; }

	const RemoteConnection& operator*() const  { return *mRemoteConnection; }
	RemoteConnection& operator*()  { return *mRemoteConnection; }

	const RemoteConnection* get() const { return mRemoteConnection; }
	RemoteConnection* get() { return mRemoteConnection; }

	/**
	 * RemoteConnectionProxy is valid if its RemoteConnectionPool and RemoteConnection
	 * pointers are valid and the referred pool contains the referred connection.
	 */
	bool valid() const;

	bool operator== (const RemoteConnection* other) const { return mRemoteConnection == other; }

private:
	// only the RemoteConnectionPool has access and can create a valid instance
	RemoteConnectionProxy(RemoteConnectionPool*, RemoteConnection*);

private:
	RemoteConnectionPool* mRemoteConnectionPool;
	RemoteConnection* mRemoteConnection;
};

template <typename SerializerTag>
class IsTransparentSerializable<RemoteConnectionProxy, SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

/**
 * Owner of every RemoteConnection.
 * It holds 2 connection maps.
 * The functions RemoteConnectionProxy#createRemoteOutgoingConnection and RemoteConnectionProxy#createRemoteIncomingConnection
 * construct a RemoteConnection, store it in RemoteConnectionPool#mRemoteConnections and
 * return a \ref RemoteConnectionProxy object for that connection.
 * If the proxy object is destroyed, the connection is moved to RemoteConnectionPool#mStoppedConnections. From there, the
 * connection can be destroyed by releaseStoppedConnections. This is called in RemoteModule#process.
 */
class RemoteConnectionPool
{
	friend RemoteConnectionProxy;
public:
	/**
	 * Function that constructs a RemoteOutgoingConnection, stores that connection in \ref mRemoteConnections and
	 * returns a \ref RemoteConnectionProxy.
	 * @param address Connection details of the outgoing connection
	 * @return proxy for the constructed RemoteConnection
	 */
	RemoteConnectionProxy createRemoteOutgoingConnection(const KnownFramework& address);

	/**
	 * Function that constructs a RemoteIncomingConnection, stores that connection in \ref mRemoteConnections and
	 * returns a \ref RemoteConnectionProxy.
	 * @return proxy for the constructed RemoteConnection
	 */
	RemoteConnectionProxy createRemoteIncomingConnection();

	/**
	 * Clears \ref mStoppedConnections to destroy the RemoteConnection objects.
	 */
	void releaseStoppedConnections();

protected:
	/**
	 * Mutex for access to \ref mRemoteConnections and \ref mStoppedConnections.
	 */
	mutable boost::recursive_mutex mConnectionsMutex;

	/**
	 * Moves the ownership for a connection from \ref mRemoteConnections to \ref mStoppedConnections.
	 * Is called in the destructor of RemoteConnectionProxy.
	 * \param connection pointer to a \ref RemoteConnection
	 */
	void stopConnection(RemoteConnection* connection);

	/**
	 * Connections for which a RemoteConnectionProxy exists
	 */
	std::set<RemoteConnection*> mRemoteConnections;

	/**
	 * Connections for which no RemoteConnectionProxy exists anymore (awaiting deletion).
	 */
	std::vector<std::unique_ptr<RemoteConnection>> mStoppedConnections;
};


///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
