/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelBuffer.h
 *    Classes for managing a channels internal buffer (slots).
 *
 * @author Erik Einhorn
 * @date   2010/09/17
 */

#ifndef _MIRA_CHANNELBUFFER_H_
#define _MIRA_CHANNELBUFFER_H_

#include <string>
#include <list>
#include <type_traits>

#ifndef Q_MOC_RUN
#include <boost/noncopyable.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/shared_mutex.hpp>
#endif

#include <error/LoggingCore.h>

#include <factory/TypeId.h>

#include <serialization/BinarySerializer.h>
#include <serialization/JSONSerializer.h>
#include <serialization/XMLSerializer.h>
#include <serialization/MetaSerializer.h>

#include <utils/Time.h>
#include <utils/Stamped.h>
#include <platform/Typename.h>

#include <math/Eigen.h>

#include <fw/FrameworkExports.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// forward decl.
template <typename TargetType>
struct ChannelBufferPromoterCommon;

/// forward decl.
template <typename TargetType>
struct ChannelBufferPromoter;

///////////////////////////////////////////////////////////////////////////////

/**
 * Mode that is used to determine the slot obtained
 * from a channel when no slot exists at the exact timestamp requested
 */
enum SlotQueryMode
{
	/// The first slot with timestamp < requested will be chosen
	OLDER_SLOT = 0,

	/// The first slot with timestamp > requested will be chosen
	NEWER_SLOT,

	/// The slot with smallest time difference to the requested will be chosen
	NEAREST_SLOT
};

/**
 * Mode that is used to determine what slots should be added to the interval
 * when not enough slots are available
 */
enum IntervalFillMode
{
	/// Prefer filling the interval with slots with timestamp < requested
	PREFER_OLDER = 0,
	/// Prefer filling the interval with slots with timestamp > requested
	PREFER_NEWER
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class that manages the slots of channels by providing read and write
 * access to them. It also secures the time order of the slots for reading
 * intervals.
 *
 * Design decisions:
 *
 *  - cannot use std::list, since (C++ STL standard) std::list::splice
 *    invalidates iterators, hence we must implement our own list.
 *    (in practice SGI implementation doesn't but Visual C++ version does)
 *  - iterators in our implementation are then pointers to ChannelBufferSlot
 *    which are always valid
 */
class MIRA_FRAMEWORK_EXPORT ChannelBufferBase : boost::noncopyable
{
	template <typename TargetType>
	friend struct ChannelBufferPromoterCommon;

	template <typename TargetType>
	friend struct ChannelBufferPromoter;

public:

	/// forward declaration
	struct Slot;

	/**
	 * Implements double-linked list item
	 */
	struct ListItem : boost::noncopyable
	{
		ListItem() {
			// creates empty list
			next = (Slot*)this;
			prev = (Slot*)this;
		}

		// both will be locked by mutex in ChannelBuffer
		Slot* next;
		Slot* prev;
	};

	/**
	 * Container for storing a single data element in the linked list.
	 */
	struct Slot : public ListItem
	{
		boost::shared_mutex lock; // locks the following data

		/**
		 * Lock for writing serializedValue in readSerializedData.
		 * It is needed, since readSerialized() can be called if the above
		 * mutex (lock) is shared locked only.
		 */
		boost::mutex serializedValueLock;

		/**
		 * Pointer to the actual timestamp in the slot data for easy access.
		 */
		const Time* timestampPtr;

		/**
		 * The data of this slot as serialized data.
		 * If empty, the data was not serialized yet.
		 */
		Buffer<uint8> serializedValue;

		/**
		 * Used to store internal flags that are especially used
		 * by the remote capability of the framework.
		 * These flags are for internal use only and are managed by
		 * the ChannelBuffer.
		 */
		uint32 flags;

		const Time& timestamp() const {
			assert(timestampPtr!=NULL); // pointer is set in ChannelBuffers allocateSlot
			return *timestampPtr;
		}
	};

public:

	/**
	 * Constructor. !KEEP INLINE! out of performance reasons.
	 * Creates an empty buffer with a maximum of 100 slots and an empty
	 * storage duration - meaning a history of 0
	 */
	ChannelBufferBase() :
		mStorageDuration(Duration::seconds(0)),
		mAutoIncreaseStorageDuration(true),
		mMinSlots(0),
		mMaxSlots(100),
		mSize(0)
	{
	}

	virtual ~ChannelBufferBase();

public:

	/**
	 * Copies the parameters of other to this.
	 */
	void cloneParameters(const ChannelBufferBase& other)
	{
		mMinSlots = other.mMinSlots;
		mMaxSlots = other.mMaxSlots;
		mStorageDuration = other.mStorageDuration;
		mAutoIncreaseStorageDuration = other.mAutoIncreaseStorageDuration;
	}

public:

	/**
	 * Sets the desired storage duration (the history).
	 * This method can increase the duration only.
	 * If the specified duration is smaller than a previous set duration
	 * the call will be ignored.
	 * @note Duration can only be increased until the maximum number of slots
	 *       is reached. Use setMaxSlots() to increase maximum number of slots.
	 */
	void setStorageDuration(const Duration& storageDuration);


	/**
	 * Sets whether the storage duration is automatically increased on read accesses.
	 * If true, any read access specifying a timestamp will automatically increase
	 * the storage duration to the time difference between the timestamp and the most recent
	 * data (trying to ensure data is kept long enough for further such requests).
	 * The default is true.
	 */
	void setAutoIncreaseStorageDuration(bool increase);

	/**
	 * Sets the desired min. number of slots for storing the history
	 * of a channel. This method can increase the min. number of slots only.
	 * Instead of specifying the min. number of slots you can also specify
	 * the desired storage duration using the setStorageDuration() method.
	 * @note The desired number of slots can be increased until the maximum
	 *       number of slots is reached. Use setMaxSlots() to increase maximum
	 *       number of slots.
	 */
	void setMinSlots(std::size_t minSlots);


	/**
	 * Sets the max. number of slots that are used for storing the history
	 * of a channel. maxSlots must at least be 1.
	 * The default is 100.
	 * @note There will only be as many slots as needed depending on the number
	 *       of publishers and subscribers. The number of slots will also
	 *       grow (history) if a storage duration is specified
	 *       via setStorageDuration()
	 */
	void setMaxSlots(std::size_t maxSlots);

public:

	/// Returns the size of the buffer
	std::size_t getSize() const;

	/// Returns the size of the max number of slots.
	std::size_t getMaxSlots() const;

	/// Returns the size of the min number of slots.
	std::size_t getMinSlots() const;

	/// Returns the storage duration.
	Duration getStorageDuration() const;

	/// Returns whether the storage duration is automatically increased.
	bool isAutoIncreasingStorageDuration() const;


public:

	/// Returns typeid of the slot values, the allocator creates.
	virtual int getTypeId() const = 0;

	/// Returns true, if the channel is typed and false, if it is untyped.
	virtual bool isTyped() const = 0;

	/// Returns true if the channel has a polymorphic type
	virtual bool isPolymorphic() const = 0;

	/**
	 * Returns the Typename of the slot values, in contrast to getTypeId() the
	 * Typename is portable and unique between different processes and operating
	 * systems.
	 */
	virtual Typename getTypename() const = 0;

	/**
	 * Sets the Typename of the slot values.
	 */
	virtual void setTypename(const Typename& name) = 0;

	/**
	 * Creates meta information for data in slot and stores all
	 * meta information in the given database
	 */
	virtual TypeMetaPtr createTypeMeta(Slot* s, MetaTypeDatabase& ioDB) = 0;

	/**
	 * Returns the meta information for the slots type.
	 */
	TypeMetaPtr getTypeMeta() const { return mTypeMeta; }

	/**
	 * Sets the meta information of the slot's type.
	 */
	void setTypeMeta(TypeMetaPtr meta) { mTypeMeta = meta; }

	/**
	 * Calling this method will fix the type of the ChannelBuffer. This
	 * method currently is used by the polymorphic buffer only. It will
	 * set the type of the buffer to the type that the buffer currently has,
	 * hence it fixates the type of the buffer and the buffer can not be
	 * promoted to any other type of a derived class.
	 */
	virtual void fixateType() = 0;

	/// Creates and returns a new slot
	virtual Slot* allocateSlot() = 0;

	/// Called to destroy the slot and free the used memory
	virtual void freeSlot(Slot* s) = 0;

	/**
	 * Returns a reference to the stamped header information.
	 */
	virtual StampedHeader& getStampedHeader(Slot* s) = 0;

	/**
	 * Returns a buffer containing the binary serialized data of the slot.
	 * The serialized data does NOT contain any StampedHeader information.
	 */
	virtual const Buffer<uint8>& readSerializedValue(Slot* s) = 0;

	/**
	 * Returns a buffer containing the binary serialized data of the slot.
	 * The serialized data does NOT contain any StampedHeader information.
	 *
	 * @param formatVersion The binary format version requested
	 */
	virtual const Buffer<uint8>& readSerializedValue(Slot* s,
	                                                 uint8 formatVersion, bool orLower) = 0;

	/**
	 * Returns a buffer containing the binary serialized data of the slot.
	 * The serialized data does NOT contain any StampedHeader information.
	 * For serialization it will use the specified codecs.
	 */
	virtual Buffer<uint8> readSerializedValue(Slot* s,
	                                          std::list<BinarySerializerCodecPtr>& codecs) = 0;

	/**
	 * Returns a buffer containing the binary serialized data of the slot.
	 * The serialized data does NOT contain any StampedHeader information.
	 * For serialization it will use the specified codecs.
	 *
	 * @param formatVersion The binary format version requested
	 */
	virtual Buffer<uint8> readSerializedValue(Slot* s,
	                                          std::list<BinarySerializerCodecPtr>& codecs,
	                                          uint8 formatVersion, bool orLower) = 0;

	/**
	 * Write the given buffer (that contains binary serialized data
	 * WITHOUT the StampedHeader) to the slot.
	 */
	virtual void writeSerializedValue(Slot* s, Buffer<uint8> data) = 0;

	/**
	 * Reads data of that slot as json representation using
	 * a default-instantiated JSONSerializer.
	 * @throw XNotImplemented if not supported by the channel
	 * @note This method is only supported by typed channels or untyped channels with
	 * valid metadata.
	 */
	virtual void readJSON(Slot* s, JSONValue& oValue) = 0;

	/**
	 * Reads data of that slot as json representation using the provided JSONSerializer.
	 * @throw XNotImplemented if not supported by the channel
	 * @note This method is only supported by typed channels or untyped channels with
	 * valid metadata.
	 */
	virtual void readJSON(Slot* s, JSONValue& oValue, JSONSerializer& serializer) = 0;

	/**
	 * Writes data in json representation into the slot, using a
	 * default-instantiated JSONDeserializer.
	 * @throw XNotImplemented if not supported by the channel
	 * @note That this method is only supported by typed channels or untyped channels with
	 * valid metadata.
	 */
	virtual void writeJSON(Slot* s, const JSONValue& value) = 0;

	/**
	 * Writes data from an initialized json deserializer into the slot.
	 * @throw XNotImplemented if not supported by the channel
	 * @note That this method is only supported by typed channels or untyped channels with
	 * valid metadata.
	 */
	virtual void writeJSON(Slot* s, JSONDeserializer& value) = 0;

	/**
	 * Writes data in xml representation into the slot, using a
	 * default-instantiated XMLDeserializer and the name of the provided node.
	 * @throw XNotImplemented if not supported by the channel
	 * @note That this method is only supported by typed channels
	 */
	virtual void writeXML(Slot* s, const XMLDom::const_iterator& node) = 0;

public:

	///////////////////////////////////////////////////////////////////////////
	// write access

	/**
	 * Returns a slot for writing, where the shared_mutex is already write locked.
	 *
	 * Implementation details:
	 * \code
	 * Cycle of a slot:
	 *
	 *    newly created (requestWriteSlot, Step ..)
	 *     |
	 *     v
	 *    writing list
	 *     |finish    |
	 *     v          |discard
	 *    ring buffer |
	 *     |read      |
	 *     v          v
	 *    mWaitingOrFree
	 *     |
	 *     v
	 * \endcode
	 *
	 */
	Slot* requestWriteSlot();

	/**
	 * Adds written slot into buffer at the right time position.
	 * Returns true, if the new Slot value is now the newest value
	 * in the buffer.
	 * dropped is set to true, if a slot with the same timestamp already
	 * exists and hence the slot was discarded.
	 * @note The lock of the slot is unlocked.
	 */
	bool finishWriteSlot(Slot* n, bool* dropped=NULL);

	/**
	 * Discards written slot
	 * @note The lock of the slot is unlocked.
	 */
	void discardWriteSlot(Slot* n);


	///////////////////////////////////////////////////////////////////////////
	// read access

	/**
	 * Returns the newest(latest) slot for reading, where the shared_mutex
	 * is already read locked.
	 * Returns NULL if no slot for reading is available.
	 */
	Slot* readNewestSlot();

	/**
	 * Returns a slot at the given time according to mode.
	 * Returns NULL if no slot exists.
	 * See mira::Channel::read() for more informations.
	 * Increases storage duration (history) automatically to serve future requests.
	 */
	Slot* readSlotAtTime(const Time& timestamp, SlotQueryMode mode);

	/**
	 * Reads an interval of slots. See mira::Channel::readInterval for more information.
	 * Increases storage duration (history) automatically to serve future requests.
	 * @throw XInvalidRead if not enough data slots available in channel
	 */
	void readInterval(const Time& timestamp, std::size_t nrSlots,
	                  std::size_t olderSlots, std::size_t newerSlots,
	                  IntervalFillMode fillMode, std::list<Slot*>& oSlots);

	/**
	 * Reads an interval of slots. See mira::Channel::readInterval for more informations.
	 * Increases storage duration (history) automatically to serve future requests.
	 * @throw XInvalidParameter if from > to
	 */
	void readInterval(const Time& from, const Time& to, std::list<Slot*>& oSlots);

public:

	/**
	 * Promotes this buffer to the specified target type (if required).
	 * This method may create and return a new ChannelBuffer of the desired
	 * TargetType. In this case the original buffer should be destroyed by
	 * the caller.
	 * In some other cases when no new buffer needs to be created the method
	 * returns a pointer to this buffer itself.
	 */
	template<typename TargetType>
	ChannelBufferBase* promote();

private:

	/**
	 * Returns the first slot in the buffer, that is older
	 * than time, or end if all slots in the buffer are newer
	 * @note mMutex must be locked, when calling this method !!!
	 */
	Slot* findFirstOlderSlot(const Time& timestamp, Slot* start=NULL);

private:

	/**
	 * Moves the successive items first..last (including from and to)
	 * and appends them after position.
	 */
	void splice(Slot* first, Slot* last, Slot* position) const;

	/**
	 * Inserts item after the specified position
	 * and removes it from its previous list, if any.
	 */
	void splice(Slot* item, Slot* position) const {
		splice(item,item,position);
	}

	static bool isEmpty(const ListItem& list) { return list.next == &list; }
	static Slot* begin(ListItem& list) { return list.next; }
	static Slot* last(ListItem& list)  { return list.prev; }
	static Slot* end(ListItem& list)   { return (Slot*)&list; }

	/// Deletes all slots in the specified list
	void deleteSlots(ListItem& list);

protected:

	/// Deletes all slots in all lists and therefore clears the whole buffer.
	void clear();

private:

	/**
	 * Resets content of the slot and prepares it for it's
	 * next usage as new slot.
	 */
	void resetSlot(Slot* s);

	// parameters:
	Duration  mStorageDuration;
	bool mAutoIncreaseStorageDuration;
	std::size_t mMinSlots;
	std::size_t mMaxSlots;


	// mutex that locks all actions when we change our lists
	mutable boost::mutex mMutex;

	std::size_t mSize; // current number of slots we are using

	ListItem mBuffer;

	/**
	 * Elements that were taken out of the ring buffer since they were
	 * still blocked by read locks when we needed to write into them
	 * again.
	 * Hence we skipped them. To avoid disrupting the chronological order
	 * we had to remove them from the current ring buffer and inserted
	 * them into the list of items where we are waiting for their unlock.
	 *
	 * This list also contains slots that were released and discarded
	 * (and hence not inserted into the ring buffer). Free elements come
	 * first.
	 */
	ListItem mWaitingOrFree;

	/**
	 * This list contains slots that are currently writing.
	 */
	ListItem mWriting;

	TypeMetaPtr mTypeMeta;

private:

	std::string dbgDumpList(ListItem& list, bool brief);

public:

	void dbgDump(const std::string& prefix, bool brief=true);

	static void dbgCheckListIntegrity(const ListItem& list);
	void dbgCheckIntegrity();
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for typed channel buffers
 */
template <typename T>
class TypedChannelBufferBase : public ChannelBufferBase
{
public:

	typedef Stamped<T> ValueType;

public:

	virtual ~TypedChannelBufferBase() { this->clear(); }

public:

	/// Typed slot derived from ChannelBufferBase::Slot
	struct Slot : public ChannelBufferBase::Slot {
		/// the actual data value, that is stored in this channel's slots
		ValueType data;
		EIGEN_MAKE_ALIGNED_OPERATOR_NEW
	};

public:

	/**
	 * Cast an abstract slot to a typed slot
	 */
	static Slot* castSlot(ChannelBufferBase::Slot* s)
	{
		// cast abstract ChannelBufferBase::Slot to TypedChannelBufferBase<T>::Slot.
		// this is safe since all slots were created by us in allocateSlot()
		// as TypedChannelBufferBase<T>::Slots
		return static_cast<Slot*>(s);
	}

public:
	/**
	 * @name Implementation of ChannelBufferBase
	 */
	//@{
	virtual int getTypeId() const { return typeId<T>(); }
	virtual bool isTyped() const { return true; }
	virtual bool isPolymorphic() const {return false; }
	virtual Typename getTypename() const { return typeName<T>(); }
	virtual void setTypename(const Typename& name) {
		MIRA_THROW(XRuntime, "Cannot set Typename for typed channels");
	}
	virtual TypeMetaPtr createTypeMeta(ChannelBufferBase::Slot* s, MetaTypeDatabase& ioDB) {
		MetaSerializer ms(ioDB);
		Slot* slot = castSlot(s);
		TypeMetaPtr meta = ms.addMeta(slot->data.internalValueRep());
		return meta;
	}

	virtual void fixateType() {} // doesn't do anything
	virtual ChannelBufferBase::Slot* allocateSlot() {
		Slot* slot = new Slot;
		slot->timestampPtr = &(static_cast<StampedHeader&>(slot->data).timestamp); // "connect" the timestamp ptr
		return slot;
	}
	virtual void freeSlot(ChannelBufferBase::Slot* s) {
		// casting s to our slot type ensures that the correct destructor is called
		Slot* slot = castSlot(s);
		delete slot;
	}

public:

	virtual StampedHeader& getStampedHeader(ChannelBufferBase::Slot* s) {
		// we can safely cast here, since s is an object of our Slot type
		Slot* slot = static_cast<Slot*>(s);
		return stampedHeader(slot->data);
	}

	virtual const Buffer<uint8>& readSerializedValue(ChannelBufferBase::Slot* s)
	{
		// lock the serialization mutex to protect serializedValueLock
		// (we or someone else may write it here)
		boost::mutex::scoped_lock lock(s->serializedValueLock);

		// no serialized data yet, so serialize it now (lazy evaluation)
		if(s->serializedValue.empty()) {
			// we can safely cast here, since s is an object of our Slot type
			Slot* slot = static_cast<Slot*>(s);

			// serialize the data
			BinaryBufferSerializer bs(&s->serializedValue);
			bs.serialize(slot->data.internalValueRep(), false);
		}

		return s->serializedValue;
	}

	
	virtual const Buffer<uint8>& readSerializedValue(ChannelBufferBase::Slot* s,
	                                                 uint8 formatVersion, bool orLower)
	{
		uint8 serializerVersion = BinaryBufferSerializer::getSerializerFormatVersion();

		if (formatVersion == serializerVersion)
			return readSerializedValue(s);

		if (formatVersion == 0) {
			// lock the serialization mutex to protect serializedValueLock
			// (we or someone else may write it here)
			boost::mutex::scoped_lock lock(s->serializedValueLock);

			// no serialized data yet, so serialize it now (lazy evaluation)
			if(s->serializedValue.empty()) {
				// we can safely cast here, since s is an object of our Slot type
				Slot* slot = static_cast<Slot*>(s);

				// serialize the data
				BinaryBufferSerializerLegacy bs(&s->serializedValue);
				bs.serialize(slot->data.internalValueRep(), false);
			}

			return s->serializedValue;
		}

		MIRA_THROW(XIO, "Requested serialized data of binary format version " << (int)formatVersion
		             << ". Only implemented for versions 0, " << (int)serializerVersion << ".");
	}

	virtual Buffer<uint8> readSerializedValue(ChannelBufferBase::Slot* s,
	                                          std::list<BinarySerializerCodecPtr>& codecs)
	{
		Buffer<uint8> serializedValue;

		// we can safely cast here, since s is an object of our Slot type
		Slot* slot = static_cast<Slot*>(s);

		BinaryBufferSerializer bs(&serializedValue);

		// register codecs
		foreach(BinarySerializerCodecPtr codec, codecs)
			bs.registerCodec(codec);

		// serialize the data
		bs.serialize(slot->data.internalValueRep(),false);

		return std::move(serializedValue);
	}

	virtual Buffer<uint8> readSerializedValue(ChannelBufferBase::Slot* s,
	                                          std::list<BinarySerializerCodecPtr>& codecs,
	                                          uint8 formatVersion, bool orLower)
	{
		uint8 serializerVersion = BinaryBufferSerializer::getSerializerFormatVersion();

		if (formatVersion == serializerVersion)
			return readSerializedValue(s, codecs);

		if (formatVersion == 0) {
			Buffer<uint8> serializedValue;

			// we can safely cast here, since s is an object of our Slot type
			Slot* slot = static_cast<Slot*>(s);

			BinaryBufferSerializerLegacy bs(&serializedValue);

			// register codecs
			foreach(BinarySerializerCodecPtr codec, codecs)
				bs.registerCodec(codec);

			// serialize the data
			bs.serialize(slot->data.internalValueRep(),false);

			return std::move(serializedValue);
		}

		MIRA_THROW(XIO, "Requested serialized data of binary format version " << (int)formatVersion
		             << ". Only implemented for versions 0, " << (int)serializerVersion << ".");
	}

	virtual void writeSerializedValue(ChannelBufferBase::Slot* s, Buffer<uint8> data)
	{
		s->serializedValue = std::move(data);

		// we can safely cast here, since s is an object of our Slot type
		Slot* slot = static_cast<Slot*>(s);

		// deserialize into our data
		BinaryBufferDeserializer bs(&s->serializedValue);

		{
			boost::mutex::scoped_lock lock(mCodecsMutex);
			bs.setCodecs(mCodecs);
		}
		bs.deserialize(slot->data.internalValueRep(), false);
		{
			boost::mutex::scoped_lock lock(mCodecsMutex);
			mCodecs = bs.getCodecs();
		}
	}

	virtual void readJSON(ChannelBufferBase::Slot* s, JSONValue& oValue)
	{
		// we can safely cast here, since s is an object of our Slot type
		Slot* slot = static_cast<Slot*>(s);

		JSONSerializer json;
		oValue = json.serialize(slot->data.internalValueRep());
	}

	virtual void readJSON(ChannelBufferBase::Slot* s, JSONValue& oValue,
	                      JSONSerializer& serializer)
	{
		// we can safely cast here, since s is an object of our Slot type
		Slot* slot = static_cast<Slot*>(s);

		oValue = serializer.serialize(slot->data.internalValueRep());
	}

	virtual void writeJSON(ChannelBufferBase::Slot* s, const JSONValue& value)
	{
		// we can safely cast here, since s is an object of our Slot type
		Slot* slot = static_cast<Slot*>(s);

		JSONDeserializer json(value);
		json.deserialize(slot->data.internalValueRep());
	}

	virtual void writeJSON(ChannelBufferBase::Slot* s, JSONDeserializer& deserializer)
	{
		// we can safely cast here, since s is an object of our Slot type
		Slot* slot = static_cast<Slot*>(s);

		deserializer.deserialize(slot->data.internalValueRep());
	}

	void writeXML(ChannelBufferBase::Slot* s, const XMLDom::const_iterator& node) override
	{
		// we can safely cast here, since s is an object of our Slot type
		Slot* slot = static_cast<Slot*>(s);
		XMLDeserializer xml(node);
		const std::string rootTag = *node;
		xml.deserializeFromNode(rootTag.c_str(), slot->data.internalValueRep());
	}
	//@}

private:

	BinaryBufferDeserializer::CodecsMap mCodecs;
	boost::mutex mCodecsMutex;
};


///////////////////////////////////////////////////////////////////////////////

/**
 * Typed ChannelBuffer.
 */
template <typename T>
class ChannelBuffer : public TypedChannelBufferBase<T>
{};

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialization for void (Untyped ChannelBuffer), which only holds the
 * StampedHeader
 */
template <>
class MIRA_FRAMEWORK_EXPORT ChannelBuffer<void> : public ChannelBufferBase
{
public:

	typedef StampedHeader ValueType;

public:

	/// Typed slot derived from ChannelBufferBase::Slot
	struct Slot : public ChannelBufferBase::Slot {
		ValueType header;
		EIGEN_MAKE_ALIGNED_OPERATOR_NEW
	};

public:

	/**
	 * Cast an abstract slot to a typed slot
	 */
	static Slot* castSlot(ChannelBufferBase::Slot* s)
	{
		// cast abstract ChannelBufferBase::Slot to TypedChannelBufferBase<T>::Slot.
		// this is safe since all slots were created by us in allocateSlot()
		// as TypedChannelBufferBase<T>::Slots
		return static_cast<Slot*>(s);
	}

public:

	ChannelBuffer() : mTypename("") {}

	virtual ~ChannelBuffer() { this->clear(); }


public:
	/**
	 * @name Implementation of ChannelBufferBase
	 */
	//@{
	virtual int getTypeId() const { return -1; }
	virtual bool isTyped() const { return false; }
	virtual bool isPolymorphic() const {return false; }
	virtual Typename getTypename() const { return mTypename; }
	virtual void setTypename(const Typename& name);
	virtual TypeMetaPtr createTypeMeta(ChannelBufferBase::Slot* s, MetaTypeDatabase& ioDB) {
		MIRA_THROW(XRuntime, "Cannot create type meta for untyped channels");
		// keep compiler happy
		return TypeMetaPtr();
	}
	virtual void fixateType() {} // doesn't do anything

	virtual ChannelBufferBase::Slot* allocateSlot() {
		Slot* slot = new Slot;
		slot->timestampPtr = &slot->header.timestamp; // "connect" the timestamp ptr
		return slot;
	}

	virtual void freeSlot(ChannelBufferBase::Slot* s) {
		// casting s to our slot type ensures that the correct destructor is called
		Slot* slot = castSlot(s);
		delete slot;
	}

public:

	virtual StampedHeader& getStampedHeader(ChannelBufferBase::Slot* s) {
		// casting here is not safe, if we actually are a typed buffer,
		// however, if we are, then the TypedChannelBuffer::getStampedHeader()
		// of that class would have been called instead of us.
		Slot* slot = static_cast<Slot*>(s);
		return slot->header;
	}

	virtual const Buffer<uint8>& readSerializedValue(ChannelBufferBase::Slot* s)
	{
		return s->serializedValue;
	}

	virtual const Buffer<uint8>& readSerializedValue(ChannelBufferBase::Slot* s,
	                                                 uint8 formatVersion, bool orLower)
	{
		BinaryBufferIstream stream(&(s->serializedValue));
		uint8 dataVersion = BinaryBufferDeserializer::getDataFormatVersion(stream);
		if ((!orLower && (dataVersion != formatVersion)) || (dataVersion > formatVersion)) {
			MIRA_THROW(XIO, "Untyped channel contains serialized data of version " << (int)dataVersion
			                << ", cannot return data of version " << (int)formatVersion << ".");
		}

		return readSerializedValue(s);
	}


	virtual Buffer<uint8> readSerializedValue(ChannelBufferBase::Slot* s,
	                                          std::list<BinarySerializerCodecPtr>& codecs)
	{
		// we must ignore the codecs
		return s->serializedValue;
	}

	virtual Buffer<uint8> readSerializedValue(ChannelBufferBase::Slot* s,
	                                          std::list<BinarySerializerCodecPtr>& codecs,
	                                          uint8 formatVersion, bool orLower)
	{
		// we must ignore the codecs
		return readSerializedValue(s, formatVersion, orLower);
	}

	virtual void writeSerializedValue(ChannelBufferBase::Slot* s, Buffer<uint8> data)
	{
		s->serializedValue = std::move(data);
	}

	virtual void readJSON(ChannelBufferBase::Slot* s, JSONValue& oValue);
	virtual void readJSON(ChannelBufferBase::Slot* s, JSONValue& oValue, JSONSerializer& serializer);

	virtual void writeJSON(ChannelBufferBase::Slot* s, const JSONValue& value);
	virtual void writeJSON(ChannelBufferBase::Slot* s, JSONDeserializer& deserializer);
	void writeXML(ChannelBufferBase::Slot* s, const XMLDom::const_iterator& node) override;

	//@}

private:

	Typename mTypename; ///< The typename of the slots
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A channel buffer for polymorphic types (classes derived from mira::Object)
 */
class MIRA_FRAMEWORK_EXPORT PolymorphicChannelBuffer : public TypedChannelBufferBase<Object*>
{
	typedef TypedChannelBufferBase<Object*> Base;

public:

	/// Constructor
	PolymorphicChannelBuffer() :
		mMostDerivedClass(NULL),
		mFixatedClass(NULL) {}

public:
	/**
	 * @name Implementation of TypedChannelBufferBase<Object*>
	 */
	//@{
	virtual bool isPolymorphic() const {return true; }

	virtual int getTypeId() const;

	virtual Typename getTypename() const;

	virtual ChannelBufferBase::Slot* allocateSlot();

	virtual void freeSlot(ChannelBufferBase::Slot* s);

	virtual void fixateType();

	virtual void writeSerializedValue(ChannelBufferBase::Slot* s, Buffer<uint8> data);
	//@}

	/// Promotes the polymorphic type of the channel to the given class type
	void promote(const Class* promotionClass);

protected:

	const Class* mMostDerivedClass;
	const Class* mFixatedClass;
};

#ifndef MIRA_WINDOWS
// declare template extern to speed up compilation time by several seconds
extern template class TypedChannelBufferBase<Object*>;
#endif

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialization for polymorphic types (only classes derived from mira::Object
 * are supported!)
 */
template <typename T>
class ChannelBuffer<T*> : public PolymorphicChannelBuffer
{
	static_assert(std::is_base_of<mira::Object,T>::value,
	              "In channels you can only use pointers of polymorphic "
	              "classes that are derived from mira::Object. Pointers to "
	              "other classes cannot be stored in a channel.");
public:
	typedef Stamped<T*> ValueType;

public:
	ChannelBuffer() {
		mMostDerivedClass = &T::CLASS();
	}
};

///////////////////////////////////////////////////////////////////////////////
// Channel buffer promotion code

/**
 * Class with a single static commonPromote method that promotes an arbitrary
 * ChannelBuffer to a ChannelBuffer of the TargetType.
 * This common method is used by the ChannelBufferPromoter classes.
 * @see ChannelBufferPromoter
 */
template <typename U>
struct ChannelBufferPromoterCommon
{
	static ChannelBufferBase* promoteUntyped(ChannelBufferBase* buffer)
	{
		assert(!buffer->isTyped() &&
		       "Can promote untyped channels to typed channels only");

		if(!buffer->getTypename().empty() && buffer->getTypename()!=typeName<U>())
			MIRA_THROW(XBadCast, "Invalid promotion from untyped to typed "
			           "ChannelBuffer. Typename does not match. ('" <<
			           buffer->getTypename() << "' != '" << typeName<U>() << "'");

		typedef ChannelBufferBase::Slot Slot;

		// create a new typed buffer
		ChannelBuffer<U>* typedBuffer = new ChannelBuffer<U>;

		// copy all parameters
		typedBuffer->cloneParameters(*buffer);

		// copy all the serialized data from the untyped buffer into the
		// typed buffer and deserialize it into the value

		// now lock the whole buffer for reading ...
		for(Slot* s=ChannelBufferBase::begin(buffer->mBuffer);
				s!=ChannelBufferBase::end(buffer->mBuffer); s=s->next)
			s->lock.lock_shared();

		// ... and deserialize the content of each slot into the new buffer
		for(Slot* s=ChannelBufferBase::begin(buffer->mBuffer);
				s!=ChannelBufferBase::end(buffer->mBuffer); s=s->next)
		{
			Slot* n = typedBuffer->requestWriteSlot(); // obtain a new slot

			// ... copy the headers
			typedBuffer->getStampedHeader(n) = buffer->getStampedHeader(s);

			// ... and write the value into the new slot (this will deserialize the data there)
			typedBuffer->writeSerializedValue(n, s->serializedValue);

			typedBuffer->finishWriteSlot(n);
		}

		// free the read lock from above ...
		for(Slot* s=ChannelBufferBase::begin(buffer->mBuffer);
		    s!=ChannelBufferBase::end(buffer->mBuffer); s=s->next)
			s->lock.unlock_shared();

		return typedBuffer;
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Class with a single static promote method that promotes an arbitrary
 * ChannelBuffer to a ChannelBuffer of the U.
 * The class is needed since C++ does not allow partial function template
 * specialization. Otherwise defining specializations of the
 * ChannelBuffer::promote() method would be much nicer (regarding access
 * rights etc.), but does not work as said before.
 * To be able to access the internals of ChannelBuffer this class is a friend
 * of ChannelBuffer.
 */
template <typename U>
struct ChannelBufferPromoter : public ChannelBufferPromoterCommon<U>
{
	typedef ChannelBufferPromoterCommon<U> Base;
	static ChannelBufferBase* promote(ChannelBufferBase* buffer)
	{
		if(buffer->isTyped())
			return buffer;  // we are promoted already, do nothing
		return Base::promoteUntyped(buffer);
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialization of ChannelBufferPromoter for polymorphic pointer channel
 * buffers.
 */
template <typename U>
struct ChannelBufferPromoter<U*>
{
	typedef ChannelBufferPromoterCommon<U> Base;
	static ChannelBufferBase* promote(ChannelBufferBase* buffer)
	{
		// if untyped, promote to typed first
		if(!buffer->isTyped())
			buffer = Base::promoteUntyped(buffer);

		PolymorphicChannelBuffer* polymorphicBuffer = dynamic_cast<PolymorphicChannelBuffer*>(buffer);

		assert(polymorphicBuffer!=NULL &&
		       "We should never reach here if the buffer to promote is not "
		       "polymorphic");

		// get class of U
		const Class* classU = &U::CLASS();

		// promote us (this will do error checking and might throw XBadCast)
		polymorphicBuffer->promote(classU);

		return polymorphicBuffer;
	}
};

///////////////////////////////////////////////////////////////////////////////

template<typename U>
ChannelBufferBase* ChannelBufferBase::promote()
{
	return ChannelBufferPromoter<U>::promote(this);
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
