/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AuthorityManager.h
 *    Manager class for all authorities in a framework.
 *
 * @author Erik Einhorn
 * @date   2010/10/07
 */

#ifndef _MIRA_AUTHORITYMANAGER_H_
#define _MIRA_AUTHORITYMANAGER_H_

#include <map>

#include <fw/Authority.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * A wrapper providing locked/shared access to an AbstractAuthority pointer
 * (used by AuthorityManager::getAuthority())
 * The lifetime of the AbstractAuthority object the pointer points to is ensured
 * as long as one wrapper instance exists that wraps the same AbstractAuthority pointer.
 * The lock is released automatically when all wrapper instances are destroyed.
 */
class SharedAuthority
{
private:
	struct Shared
	{
		Shared(AbstractAuthority* iAuthority, boost::shared_mutex& mutex) :
			authority(iAuthority),
			mLock(mutex) {}

		~Shared() {
			mLock.unlock();
		}

		AbstractAuthority* authority;
	private:
		boost::shared_lock<boost::shared_mutex> mLock;
	};

public:

	/// There is no need to call this constructor manually
	SharedAuthority(AbstractAuthority* iAuthority, boost::shared_mutex& mutex) :
		mShared(new Shared(iAuthority, mutex)) {}

	/// cast to AbstractAuthority pointer
	operator AbstractAuthority*() { return mShared->authority; }

	/// access the wrapped AbstractAuthority pointer
	AbstractAuthority* authority() { return mShared->authority; }

private:

	boost::shared_ptr<Shared> mShared;

};

/**
 * @ingroup FWModule
 * A structure used for returning the complete result of a (remote)
 * property query, including potential info about a thrown exception.
 */
struct PropertyQueryResult
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Value", value, "The property value (if query succeeded)");
		r.member("SerializedException", exception,
		         "Buffer containing the serialized exception (or empty)");
		r.member("ErrorMessage", message,
		         "The error message from the exception (or empty)");
	}

	json::Value value;         // main: the json-serialized property value (if no exception)
	Buffer<uint8> exception;   // binary-serialized exception
	std::string message;       // exception message
};

/**
 * @ingroup FWModule
 * Central instance that stores all created Authorities.
 * Each Authority will automatically register and unregister here when
 * checking in / checking out.
 */
class MIRA_FRAMEWORK_EXPORT AuthorityManager
{
public:

	AuthorityManager();

	~AuthorityManager();

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IAuthorityManager");
		r.method("hasAuthority", &AuthorityManager::hasAuthority, this,
		         "Returns if an authority with the given global id exists.",
		         "id", "authority id", "/MyAuthority");
		r.method("isLocal", &AuthorityManager::isLocal, this,
		         "Returns true if authority with given id is a local one "
		         "(The instance lives in this framework/process).",
		         "id", "authority id", "/MyAuthority");
		r.method("getAuthorities", &AuthorityManager::getAuthorities, this,
		         "Returns a list of existing authorities.");
	}

public:

	/**
	 * Adds an authority to the list of checked in authorities
	 * @throw XInvalidParameter if authority already exists
	 */
	void addAuthority(AbstractAuthority* authority);

	/**
	 * Removes an authority
	 */
	void removeAuthority(AbstractAuthority* authority);

	/**
	 * Start the authority with the given id. See AbstractAuthority::start()
	 */
	void start(const std::string& id) const;

	/**
	 * Stops the authority with the given id. See AbstractAuthority::stop()
	 */
	void stop(const std::string& id) const;

	/**
	 * Returns if an authority with the given global id exists.
	 * @return true if exists
	 */
	bool hasAuthority(const std::string& id) const;

	/**
	 * Returns true if authority  with given id is started.
	 * See AbstractAuthority::isRunning()
	 */
	bool isRunning(const std::string& id) const;

	/**
	 * Returns true if authority with given id is a local one (The instance lives
	 * in this framework/process)
	 */
	bool isLocal(const std::string& id) const;

	/**
	 * Returns true if authority with given id has an unrecoverable error.
	 * See AbstractAuthority::hasUnrecoverableFailure()
	 */
	bool hasUnrecoverableFailure(const std::string& id) const;

	/**
	 * Returns the status informations of the authority with given id.
	 * See AbstractAuthority::getStatusMap()
	 */
	StatusManager::StatusMap getStatusMap(const std::string& id) const;
	/**
	 * Returns the description of the authority with given id.
	 * See AbstractAuthority::getDescription()
	 */
	AuthorityDescription getDescription(const std::string& id) const;

	/**
	 * Returns the published channels of the authority with given id.
	 * See AbstractAuthority::getPublishedChannels()
	 */

	std::set<std::string> getPublishedChannels(const std::string& id) const;
	/**
	 * Returns the subscribed channels of the authority with given id.
	 * See AbstractAuthority::getSubscribedChannels()
	 */
	std::set<std::string> getSubscribedChannels(const std::string& id) const;

	/**
	 * Returns the properties of the authority with given id.
	 * See AbstractAuthority::getProperties()
	 */
	boost::shared_ptr<PropertyNode> getProperties(const std::string& id) const;

	/**
	 * Returns a list of existing authorities.
	 * @return list of authority names
	 */
	std::set<std::string> getAuthorities() const;

	/**
	 * Get direct access to an authority pointer. The access is protected by a wrapper
	 * object that ensures the lifetime of the Authority as long as the wrapper exists.
	 */
	SharedAuthority getAuthority(const std::string& id);

private:

	struct LockableAuthority
	{
		LockableAuthority(AbstractAuthority* a) : authority(a) {}

		AbstractAuthority* authority; /// pointer to authority object
		boost::shared_mutex mutex; /// mutex that ensures lifetime of authority as long as it is locked
	};
	typedef boost::shared_ptr<LockableAuthority> LockableAuthorityPtr;
	typedef std::map<std::string, LockableAuthorityPtr> AuthorityMap;

	/// maps from global authority ids to the lockable Authority objects
	AuthorityMap mAuthorities;

	/// mutex that protects map above
	mutable boost::mutex mAuthoritiesMutex;

private:

	/**
	 * Use this for the common pattern "find id in mAuthorities and return the found entry"
	 * (or throw XInvalidParameter if not found).
	 *
	 * mAuthoritiesMutex is kept locked only while querying mAuthorities,
	 * the specific entry's mutex is locked before unlocking mAuthoritiesMutex.
	 *
	 * The mutex is locked (shared) on the returned LockableAuthorityPtr,
	 * so the caller is responsible to adopt and unlock it!
	 * (this ensures the authority can not be removed while working on it)
	 */
	LockableAuthorityPtr findAuthority(const std::string& id) const;

private:

	// authority of the AuthorityManager (must be last member!!)
	Authority mAuthority;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
