/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelContentChange.h
 *    Filtering channel messages by only calling the callback function when the content has changed
 *
 * @author Tim Langner
 * @date   2015/09/18
 */

#ifndef _MIRA_CHANNELCONTENTCHANGE_H_
#define _MIRA_CHANNELCONTENTCHANGE_H_

#include <fw/ChannelReadWrite.h>
#include <fw/Framework.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * Class that can be registered as a filter when subscribing to channels
 * to only get a callback when the content of the channel changes.
 * This requires the operator== and copy constructor to be implemented for the channels data type T.
 * Does not work with pointers. The constructor takes the function pointer to the proper callback
 * and the class' callback method is registered instead.
 *
 * \code
 * class MyUnit : public MicroUnit
 * {
 *		MyUnit() :
 *			myFilter(Duration::seconds(2)) {}
 *
 * 		void initialize()
 * 		{
 * 			myFilter.subscribe(*this, "IntChannel", &MyUnit::callback, this);
 * 		}
 *
 * 		void onCallback(ChannelRead<int> channel)
 * 		{}
 *
 * 		ChannelContentChange<int> myFilter;
 * }
 * \endcode
 */
template <typename T>
class ChannelContentChange
{
public:
	/**
	 * Call this instead of Authority::subscribe()
	 * @param authority The authority that does the basic subscription
	 * @param channelID The ID of the channel we are subscribing on
	 * @param fn The function that should be called when the filter matches
	 * @return The channel subscribed on
	 */
	Channel<T> subscribe(Authority& authority,
	                     const std::string& channelID,
	                     boost::function<void (ChannelRead<T>)> fn)
	{
		mFunction = fn;
		return authority.subscribe<T>(channelID, &ChannelContentChange<T>::callback, this);
	}

	/**
	 * Same as above but with a function and object pointer. Provided for convenience.
	 */
	template<typename Class>
	Channel<T> subscribe(Authority& authority,
	                     const std::string& channelID,
	                     void (Class::*f)(ChannelRead<T>), Class* obj)
	{
		return subscribe(authority, channelID, boost::bind(f, obj, _1));
	}

	/// Called when channel has new data
	void callback(ChannelRead<T> value)
	{
		if (!mLast || *mLast == value->value())
		mFunction(value);
		mLast = value->value();
	}

private:
	boost::function<void (ChannelRead<T>)> mFunction;
	boost::optional<T> mLast;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
