/*
 * Copyright (C) 2017 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ExtensibleEnumTest.C
 *    Test cases for ExtensibleEnum.h.
 *
 * @date 2017/10/06
 *
 * @author Christof Schröter
 */

#include <boost/test/unit_test.hpp>

#include <boost/algorithm/string/replace.hpp>

#include <utils/ExtensibleEnum.h>
#include <serialization/Print.h>

using namespace mira;

namespace A
{
	MIRA_EXTENSIBLE_ENUM_DECLARE(Enum, ExtensibleEnum<Enum>,
                                 EnumValueA, EnumValueB)

	MIRA_EXTENSIBLE_ENUM_DEFINE(Enum, EnumValueA, EnumValueB)
}

namespace B
{
	MIRA_EXTENSIBLE_ENUM_DECLARE(Enum, A::Enum,
	                             EnumValueC)

	MIRA_EXTENSIBLE_ENUM_DEFINE(B::Enum, EnumValueC)
}

BOOST_AUTO_TEST_CASE( testNonCounting )
{
	A::Enum Enum1 = A::Enum::EnumValueA;
	A::Enum Enum2 = A::Enum::EnumValueB;
	A::Enum Enum3 = B::Enum::EnumValueC;
	B::Enum Enum4 = B::Enum::EnumValueC;

	// shall not compile!
	//A::Enum Enum5;

	BOOST_CHECK(Enum1 == A::Enum::EnumValueA);
	BOOST_CHECK(Enum2 == A::Enum::EnumValueB);
	BOOST_CHECK(Enum3 == B::Enum::EnumValueC);
	BOOST_CHECK(Enum4 == B::Enum::EnumValueC);

	BOOST_CHECK(Enum1 == Enum1);
	BOOST_CHECK(Enum1 != Enum2);
	BOOST_CHECK(Enum1 != Enum3);
	BOOST_CHECK(Enum1 != Enum4);
	BOOST_CHECK(Enum2 != Enum1);
	BOOST_CHECK(Enum2 == Enum2);
	BOOST_CHECK(Enum2 != Enum3);
	BOOST_CHECK(Enum2 != Enum4);
	BOOST_CHECK(Enum3 != Enum1);
	BOOST_CHECK(Enum3 != Enum2);
	BOOST_CHECK(Enum3 == Enum3);
	BOOST_CHECK(Enum3 == Enum4);
	BOOST_CHECK(Enum4 != Enum1);
	BOOST_CHECK(Enum4 != Enum2);
	BOOST_CHECK(Enum4 == Enum3);
	BOOST_CHECK(Enum4 == Enum4);
}

namespace CountingA
{
	MIRA_EXTENSIBLE_ENUM_DECLARE(Enum, (ExtensibleEnum<Enum, false, true>),
                                 EnumValueA, EnumValueB, EnumValueC, EnumValueD)

		MIRA_EXTENSIBLE_ENUM_DEFINE(Enum,
		                            EnumValueA, EnumValueB, EnumValueC, EnumValueD)
}

namespace CountingB
{
	MIRA_EXTENSIBLE_ENUM_DECLARE(Enum, CountingA::Enum,
	                             EnumValueE, EnumValueF, EnumValueG)

	MIRA_EXTENSIBLE_ENUM_DEFINE(Enum,
	                            EnumValueE, EnumValueF, EnumValueG)
}

BOOST_AUTO_TEST_CASE( testCounting )
{
	CountingA::Enum Enum1 = CountingA::Enum::EnumValueA;
	CountingA::Enum Enum2 = CountingA::Enum::EnumValueC;
	CountingA::Enum Enum3 = CountingB::Enum::EnumValueF;
	CountingB::Enum Enum4 = CountingB::Enum::EnumValueF;

	// shall not compile!
	//CountingA::Enum Enum5;

	BOOST_CHECK(Enum1 == CountingA::Enum::EnumValueA);
	BOOST_CHECK(Enum2 == CountingA::Enum::EnumValueC);
	BOOST_CHECK(Enum3 == CountingB::Enum::EnumValueF);
	BOOST_CHECK(Enum4 == CountingB::Enum::EnumValueF);

	BOOST_CHECK(Enum1 == Enum1);
	BOOST_CHECK(Enum1 != Enum2);
	BOOST_CHECK(Enum1 != Enum3);
	BOOST_CHECK(Enum1 != Enum4);
	BOOST_CHECK(Enum2 != Enum1);
	BOOST_CHECK(Enum2 == Enum2);
	BOOST_CHECK(Enum2 != Enum3);
	BOOST_CHECK(Enum2 != Enum4);
	BOOST_CHECK(Enum3 != Enum1);
	BOOST_CHECK(Enum3 != Enum2);
	BOOST_CHECK(Enum3 == Enum3);
	BOOST_CHECK(Enum3 == Enum4);
	BOOST_CHECK(Enum4 != Enum1);
	BOOST_CHECK(Enum4 != Enum2);
	BOOST_CHECK(Enum4 == Enum3);
	BOOST_CHECK(Enum4 == Enum4);

	BOOST_CHECK(CountingA::Enum::count() == 7);
	BOOST_CHECK(CountingB::Enum::count() == 7);
}

namespace CountingLockingA
{
	MIRA_EXTENSIBLE_ENUM_DECLARE(Enum, (ExtensibleEnum<Enum, false, true, int, true>),
                                 EnumValueA, EnumValueB, EnumValueC, EnumValueD)
	MIRA_EXTENSIBLE_ENUM_DEFINE(Enum,
	                            EnumValueA, EnumValueB, EnumValueC, EnumValueD)
}

namespace CountingLockingB
{
	MIRA_EXTENSIBLE_ENUM_DECLARE(Enum, CountingLockingA::Enum,
	                             EnumValueE, EnumValueF, EnumValueG)

	MIRA_EXTENSIBLE_ENUM_DEFINE(Enum,
	                            EnumValueE, EnumValueF, EnumValueG)
}

BOOST_AUTO_TEST_CASE( testCountingLocking )
{
	CountingLockingA::Enum Enum1 = CountingLockingA::Enum::EnumValueA;
	CountingLockingA::Enum Enum2 = CountingLockingA::Enum::EnumValueC;
	CountingLockingA::Enum Enum3 = CountingLockingB::Enum::EnumValueF;
	CountingLockingB::Enum Enum4 = CountingLockingB::Enum::EnumValueF;

	// shall not compile!
	//CountingLockingA::Enum Enum5;

	BOOST_CHECK(Enum1 == CountingLockingA::Enum::EnumValueA);
	BOOST_CHECK(Enum2 == CountingLockingA::Enum::EnumValueC);
	BOOST_CHECK(Enum3 == CountingLockingB::Enum::EnumValueF);
	BOOST_CHECK(Enum4 == CountingLockingB::Enum::EnumValueF);

	BOOST_CHECK(Enum1 == Enum1);
	BOOST_CHECK(Enum1 != Enum2);
	BOOST_CHECK(Enum1 != Enum3);
	BOOST_CHECK(Enum1 != Enum4);
	BOOST_CHECK(Enum2 != Enum1);
	BOOST_CHECK(Enum2 == Enum2);
	BOOST_CHECK(Enum2 != Enum3);
	BOOST_CHECK(Enum2 != Enum4);
	BOOST_CHECK(Enum3 != Enum1);
	BOOST_CHECK(Enum3 != Enum2);
	BOOST_CHECK(Enum3 == Enum3);
	BOOST_CHECK(Enum3 == Enum4);
	BOOST_CHECK(Enum4 != Enum1);
	BOOST_CHECK(Enum4 != Enum2);
	BOOST_CHECK(Enum4 == Enum3);
	BOOST_CHECK(Enum4 == Enum4);

	BOOST_CHECK(CountingLockingA::Enum::count() == 7);
	BOOST_CHECK(CountingLockingB::Enum::count() == 7);

	// shall not compile!
	//CountingA::Enum Enum1nolock = CountingLockingA::Enum::EnumValueA;
	//BOOST_CHECK(Enum1 != Enum1nolock);
}

namespace NamedA
{
	MIRA_EXTENSIBLE_ENUM_DECLARE(Enum, (ExtensibleEnum<Enum, true>),
                                 EnumValueA, EnumValueB)
	MIRA_EXTENSIBLE_ENUM_DEFINE(Enum, EnumValueA("A::A"), EnumValueB("A::B"))
}

namespace NamedB
{
	MIRA_EXTENSIBLE_ENUM_DECLARE(Enum, NamedA::Enum,
	                             EnumValueC, EnumValueD)

	MIRA_EXTENSIBLE_ENUM_DEFINE(Enum, EnumValueC("B::C"), EnumValueD("B::C")) // same id on purpose!
}

BOOST_AUTO_TEST_CASE( testNamed )
{
	NamedA::Enum Enum1 = NamedA::Enum::EnumValueA;
	NamedA::Enum Enum2 = NamedA::Enum::EnumValueB;
	NamedA::Enum Enum3 = NamedB::Enum::EnumValueC;
	NamedB::Enum Enum4 = NamedB::Enum::EnumValueD;

	BOOST_CHECK(Enum1 == NamedA::Enum::EnumValueA);
	BOOST_CHECK(Enum2 == NamedA::Enum::EnumValueB);
	BOOST_CHECK(Enum3 == NamedB::Enum::EnumValueC);
	BOOST_CHECK(Enum4 == NamedB::Enum::EnumValueD);

	BOOST_CHECK(Enum1 == Enum1);
	BOOST_CHECK(Enum1 != Enum2);
	BOOST_CHECK(Enum1 != Enum3);
	BOOST_CHECK(Enum1 != Enum4);
	BOOST_CHECK(Enum2 != Enum1);
	BOOST_CHECK(Enum2 == Enum2);
	BOOST_CHECK(Enum2 != Enum3);
	BOOST_CHECK(Enum2 != Enum4);
	BOOST_CHECK(Enum3 != Enum1);
	BOOST_CHECK(Enum3 != Enum2);
	BOOST_CHECK(Enum3 == Enum3);
	BOOST_CHECK(Enum3 != Enum4);
	BOOST_CHECK(Enum4 != Enum1);
	BOOST_CHECK(Enum4 != Enum2);
	BOOST_CHECK(Enum4 != Enum3);
	BOOST_CHECK(Enum4 == Enum4);

	BOOST_CHECK(Enum1.id() == "A::A");
	BOOST_CHECK(Enum2.id() == "A::B");
	BOOST_CHECK(Enum3.id() == "B::C");
	BOOST_CHECK(Enum4.id() == "B::C");

	std::stringstream s;
	s << print(Enum1);
	BOOST_CHECK_EQUAL(boost::replace_all_copy(boost::replace_all_copy(s.str(), "\n", ""), " ", ""),
 	                  boost::replace_all_copy(boost::replace_all_copy( std::string("{\"id\": \"A::A\"}"), "\n", ""), " ", ""));
}

// shall not compile
//MIRA_EXTENSIBLE_ENUM_DECLARE(FloatEnum, (ExtensibleEnum<FloatEnum, true, float>),
//                             EnumValueA)
