/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationContainerImportTest.C
 *    Test for serialization framework.
 *
 * @author Christof Schröter
 */

#include <boost/test/unit_test.hpp>

#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>

#include <serialization/XMLSerializer.h>
#include <serialization/JSONSerializer.h>
#include <serialization/PropertySerializer.h>

using namespace std;
using namespace mira;

//////////////////////////////////////////////////////////////////////////////

class RecursiveStringMapTestClass
{
public:
	RecursiveStringMapTestClass() {};
	RecursiveStringMapTestClass(int i) {
		for(int j = 1; j <= i; j++)
			map[to_string((int64)j)]=j;
	}


	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("map", map, "");
	}

	std::map<string, int> map;
};

//////////////////////////////////////////////////////////////////////////////

template <typename Container>
void CHECK_EQUAL(const Container& c1, const Container& c2)
{
	BOOST_CHECK_EQUAL(c1.size(), c2.size());
	typename Container::const_iterator i1 = c1.begin(), i2 = c2.begin();

	for (; i1 != c1.end(); ++i1, ++i2)
		BOOST_CHECK_EQUAL(*i1, *i2);
}

template <typename T>
void CHECK_EQUAL(const map<T,T>& c1, const map<T,T>& c2)
{
	BOOST_CHECK_EQUAL(c1.size(), c2.size());
	typename map<T,T>::const_iterator i1 = c1.begin(), i2 = c2.begin();

	for (; i1 != c1.end(); ++i1, ++i2) {
		BOOST_CHECK_EQUAL(i1->first, i2->first);
		BOOST_CHECK_EQUAL(i1->second, i2->second);
	}
}

void CHECK_EQUAL(const map<string,RecursiveStringMapTestClass>& c1, const map<string,RecursiveStringMapTestClass>& c2)
{
	//Compare outer and inner size
	BOOST_CHECK_EQUAL(c1.size(), c2.size());
	map<string,RecursiveStringMapTestClass>::const_iterator i1 = c1.begin(), i2 = c2.begin();

	for (; i1 != c1.end(); ++i1, ++i2) {
		BOOST_CHECK_EQUAL(i1->second.map.size(), i2->second.map.size());
		auto inner_i1 = (*i1).second.map.begin(), inner_i2 = (*i2).second.map.begin();
		for(; inner_i1 != (*i1).second.map.end(); inner_i1++, inner_i2++) {
			BOOST_CHECK_EQUAL(inner_i1->first, inner_i1->first);
			BOOST_CHECK_EQUAL(inner_i1->second, inner_i1->second);
		}
	}
}

void insert(vector<int>& container, int value)
{
	container.push_back(value);
}

void insert(vector<string>& container, int& value)
{
	container.push_back(toString(value));
}

void insert(set<int>& container, int value)
{
	container.insert(value);
}

void insert(set<string>& container, int value)
{
	container.insert(toString(value));
}

void insert(map<int,int>& container, int value)
{
	container[value] = 2 * value;
}

void insert(map<string,string>& container, int value)
{
	container[toString(value)] = "-"+toString(value)+"-";
}

void insert(map<string,RecursiveStringMapTestClass>& container, int value)
{
	container[toString(value)] = RecursiveStringMapTestClass(value);
}

template<typename T, typename T2>
void writeMapStandard(const map<T,T2>& container, json::Value& value)
{
	value = json::Array();
	foreach(const auto & p, container) {
		json::Value pair = json::Object();
		pair.get_obj()["First"] = json::Value(p.first);
		pair.get_obj()["Second"] = json::Value(p.second);
		value.get_array().push_back(pair);
	}
}

template<typename T>
void writeMapStandard(const map<T,RecursiveStringMapTestClass>& container, json::Value& value)
{
	value = json::Array();
	foreach(const auto & p, container) {
		json::Value pair = json::Object();
		pair.get_obj()["First"] = json::Value(p.first);
		pair.get_obj()["Second"] = json::Object();

		json::Array inner_array;
		foreach(const auto & m, p.second.map){
			json::Value inner_pair = json::Object();
			inner_pair.get_obj()["First"] = json::Value(m.first);
			inner_pair.get_obj()["Second"] = json::Value(m.second);
			inner_array.push_back(inner_pair);
		}
		pair.get_obj()["Second"].get_obj()["map"] = inner_array;
		value.get_array().push_back(pair);
	}
}

template<typename T>
void writeMapAsObject(const map<string,T>& container, json::Value& value)
{
	value = json::Object();
	foreach(const auto & p, container) {
		value.get_obj()[p.first] = json::Value(p.second);
	}
}

void writeMapAsObject(const map<string,RecursiveStringMapTestClass>& container, json::Value& value)
{
	value = json::Object();
	foreach(const auto & p, container) {
		json::Value inner_object = json::Object();
		foreach(const auto & m, p.second.map){
			inner_object.get_obj()[m.first] = json::Value(m.second);
		}
		json::Value map = json::Object();
		map.get_obj()["map"] = inner_object;
		value.get_obj()[p.first] = map;
	}
}

template<typename T1, typename T2>
void write(const map<T1,T2>& container, json::Value& value)
{
	writeMapStandard(container, value);
}

template<typename T>
void write(const map<string,T>& container, json::Value& value, bool asObject)
{
	if (asObject)
		writeMapAsObject(container, value);
	else
		writeMapStandard(container, value);
}

template<typename T1, typename T2>
void testImportJSONMap(int length, bool pp = false)
{
	map<T1, T2> c1, c2;
	for (int n = 0; n < length; ++n)
		insert(c1, n);

	json::Value v;
	write(c1, v);

	stringstream ss;
	json::write(v, ss, pp);

	cout << ss.str() << endl;

	json::Value v2;
	json::read(ss.str(), v2);

	JSONDeserializer d(v2);

	d.deserialize(c2);
	CHECK_EQUAL(c1, c2);
}

template<typename T>
void testImportJSONStringMap(int length, bool pp = false,
                             bool writeAsObject = false,
                             bool readAsObject = false)
{
	map<std::string, T> c1, c2;
	for (int n = 0; n < length; ++n)
		insert(c1, n);

	json::Value v;
	write(c1, v, writeAsObject);

	stringstream ss;
	json::write(v, ss, pp);

	cout << ss.str() << endl;

	json::Value v2;
	json::read(ss.str(), v2);

	JSONDeserializer d(v2, readAsObject ? JSONDeserializer::ACCEPT_STRING_MAP_AS_OBJECT : JSONDeserializer::STANDARD);

	if (writeAsObject && !readAsObject) {
		BOOST_CHECK_THROW(d.deserialize(c2), XInvalidConfig);
	} else {
		d.deserialize(c2);
		CHECK_EQUAL(c1, c2);
	}
}

template<typename T>
void write(const map<T,T>& container, XMLDom& xml, bool oldFormat = false)
{
//	old format = sequence of alternating key/item elements
//	new format = sequence of pairs, with First/Second element each

	XMLDom::sibling_iterator xmlroot = xml.root();
	XMLDom::sibling_iterator node;
	XMLDom::sibling_iterator pairnode;

	node = xmlroot.add_child("test");
	foreach(const auto & p, container) {
		if (oldFormat) {
			node.add_child("key").add_content(toString(p.first));
			node.add_child("item").add_content(toString(p.second));
		} else {
			pairnode = node.add_child("item");
			pairnode.add_child("First").add_content(toString(p.first));
			pairnode.add_child("Second").add_content(toString(p.second));
		}
	}
}

template<typename T1, typename T2>
void testImportXMLMap(int length, bool oldFormat = false)
{
	// oldFormat is only used when creating the xml
	// deserialization needs to handle both and find out format by itself
	map<T1, T2> c1, c2;
	for (int n = 0; n < length; ++n)
		insert(c1, n);

	XMLDom xml;
	write(c1, xml, oldFormat);
	string sxml = xml.saveToString();

	cout << sxml << endl;

	XMLDom xml2;
	xml2.loadFromString(sxml);
	XMLDeserializer d(xml2);
	
	d.deserialize("test", c2);
	CHECK_EQUAL(c1, c2);	
}

BOOST_AUTO_TEST_CASE( TestJSONToMap )
{
	testImportJSONMap<int,int>(5, /*pp=*/false);
	testImportJSONMap<int,int>(5,        true);

	testImportJSONStringMap<string>(5, /*pp=*/false, /*writeAsObject=*/false, /*readAsObject=*/false);
	testImportJSONStringMap<string>(5,        false,                   false,                  true);
	testImportJSONStringMap<string>(5,        false,                   true,                   false);
	testImportJSONStringMap<string>(5,        false,                   true,                   true);
	testImportJSONStringMap<string>(5,        true,                    false,                  false);
	testImportJSONStringMap<string>(5,        true,                    false,                  true);
	testImportJSONStringMap<string>(5,        true,                    true,                   false);
	testImportJSONStringMap<string>(5,        true,                    true,                   true);

	testImportJSONStringMap<RecursiveStringMapTestClass>(5,        true,                    true,                   true);
	testImportJSONStringMap<RecursiveStringMapTestClass>(5,        true,                    true,                   false);
	testImportJSONStringMap<RecursiveStringMapTestClass>(5,        true,                    false,                   true);
	testImportJSONStringMap<RecursiveStringMapTestClass>(5,        true,                    false,                   false);
	testImportJSONStringMap<RecursiveStringMapTestClass>(5,        false,                    true,                   true);
	testImportJSONStringMap<RecursiveStringMapTestClass>(5,        false,                    true,                   false);
	testImportJSONStringMap<RecursiveStringMapTestClass>(5,        false,                    false,                   true);
	testImportJSONStringMap<RecursiveStringMapTestClass>(5,        false,                    false,                   false);
}


BOOST_AUTO_TEST_CASE( TestXMLToMap )
{
//	testImportXMLMap<int,int>(5); // for now we only keep the key/item sequence format for XML
	testImportXMLMap<int,int>(5, /*oldFormat=*/true);

//	testImportXMLMap<string, string>(5);
	testImportXMLMap<string, string>(5, /*oldFormat=*/true);
}

///////////////////////////////////////////////////////////////////////////////
