/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PrintTest.h
 *    Test for mira::print()
 *
 * @author Christof Schröter
 * @date   2020/09/19
 */

#include <boost/test/unit_test.hpp>

#include <serialization/Print.h>
#include <serialization/adapters/std/vector>

using namespace std;
using namespace mira;



struct PrintTestClassBasic1
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i, "");
		r.member("f", f, "");
		r.member("d", d, "");
	}

	int i;
	float f;
	double d;
};

struct PrintTestClassBasic2
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("s", s, "");
		r.member("b", b, "");
		r.member("vf", vf, "");
	}

	std::string s;
	bool b;
	std::vector<float> vf;
};

struct PrintTestClassComposed
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("el1", el1, "");
		r.member("el2", el2, "");
	}

	PrintTestClassBasic1 el1;
	PrintTestClassBasic2 el2;
};

BOOST_AUTO_TEST_CASE( PrintCompatibilityTest )
{
	PrintTestClassComposed v({{1, 32.12f, 3.333333333}, {"test", true, {-1.f, 4321.0f, 0.0001f}}});

	std::stringstream ss;

	// Test that print(v, FORMAT) variants give same result as print(v, true/false)
	ss << print(v);
	std::string pdefault = ss.str();
	ss.str("");

	ss << print(v, true);
	std::string pretty = ss.str();
	ss.str("");

	ss << print(v, false);
	std::string unpretty = ss.str();
	ss.str("");

	ss << print(v, PRETTY_PRINT, 16);
	std::string pretty_16 = ss.str();
	ss.str("");

	BOOST_CHECK_EQUAL(pdefault, pretty_16);
	BOOST_CHECK_EQUAL(pretty, pretty_16);

	ss << print(v, FORMATTED_PRINT);
	std::string formatted_default = ss.str();
	ss.str("");

	BOOST_CHECK_EQUAL(unpretty, formatted_default);

	// Test that print(v, PRETTY_FORMAT) or print(v, PRETTY_FORMAT, 16) makes a difference
	ss << print(v, PRETTY_PRINT);
	std::string pretty_default = ss.str();
	ss.str("");

	BOOST_CHECK_PREDICATE( std::not_equal_to<std::string>(),
	                       (pretty_16)(pretty_default) );

	ss << print(v, COMPACT_PRINT);
	std::string compact_default = ss.str();
	ss.str("");

	std::cout << pretty_16 << std::endl << std::endl;
	std::cout << pretty_default << std::endl << std::endl;
	std::cout << formatted_default << std::endl << std::endl;
	std::cout << compact_default << std::endl << std::endl;

	// Test that PRETTY/FOMRATTED are multi-line, COMPACT is single-line
	BOOST_CHECK_PREDICATE( std::not_equal_to<size_t>(),
	                       (pretty_16.find("\n"))(std::string::npos) );
	BOOST_CHECK_PREDICATE( std::not_equal_to<size_t>(),
	                       (formatted_default.find("\n"))(std::string::npos) );
	BOOST_CHECK_EQUAL(compact_default.find("\n"), std::string::npos);

}
