/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file EigenFormatTest.C
 *    Test cases for EigenFormat parsing code.
 *
 * @author Erik Einhorn
 * @date   2010/08/31
 */

#include <boost/test/unit_test.hpp>

#include <iostream>
#include <sstream>

#include <math/EigenFormat.h>

using namespace std;
using namespace Eigen;
using namespace mira;


BOOST_AUTO_TEST_CASE( EigenFormatTest )
{

	Matrix4f a,b;
	a << 1,2,3,4,
		 5,6,7,8,
		 9,10,11,12,
		 13,14,15,16;

	string s = "   [ 1,2,3,4;5, 6,  7,8 ;  9,10   ,11,12 ;13,14 , 15,16 ]";
	stringstream ss(s);

	// parse !
	ss >> format(b);

	BOOST_CHECK_EQUAL(a,b);

	// additional char
	s = "[1,2,3a,4;5,6,7;9,10,11,12;13,14,15,16]";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);

	// missing [
	s = "1,2,3,4;5,6,7,8;9,10,11,12;13,14,15,16]";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);

	// missing element in second row
	s = "[1,2,3,4;5,6,7;9,10,11,12;13,14,15,16]";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);

	// missing ,
	s = "[1,2 3,4;5,6,7,8;9,10,11,12;13,14,15,16]";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);

	// misssing ;
	s = "[1,2,3,4;5,6,7,8 9,10,11,12;13,14,15,16]";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);

	// missing row
	s = "[1,2,3,4;5,6,7,8;9,10,11,12]";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);

	// missing ]
	s = "[1,2,3,4;5,6,7,8;9,10,11,12;13,14,15,16";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);

	// missing end
	s = "[1,2,3,4;5,6,7,8;9,10,11,12;13,14,15,    ";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);

	// missing end
	s = " 1 2 3 4  \n 5 6  7 8\n 9 10   11 12   \n    ";
	ss.clear();
	ss.str(s);
	BOOST_CHECK_THROW(ss >> format(b), mira::XIO);


	// for testing if we do not run into an endless loop
	s = " 1 2 3 4\n5 6  7 8\n 9 10 11 12\n3 3 3 3  ";
	ss.clear();
	ss.str(s);
	ss >> format(b, EigenFormat::eigen());

}

BOOST_AUTO_TEST_CASE( EigenFormatTestDynamic )
{
	Matrix4f a;
	a << 1,2,3,4,
		 5,6,7,8,
		 9,10,11,12,
		 13,14,15,16;

	MatrixXf b;
	string s = "   [ 1,2,3,4;5, 6,  7,8 ;  9,10   ,11,12 ;13,14 , 15,16 ]";
	stringstream ss(s);

	ss >> format(b);
	BOOST_REQUIRE_EQUAL(a.rows(),b.rows());
	BOOST_REQUIRE_EQUAL(a.cols(),b.cols());
	BOOST_CHECK_EQUAL(a,b);

	b.resize(0,0);

	// eigen format
	s = " 1 2 3 4  \n 5 6  7 8\n 9 10   11 12   \n  13 14 15 16";
	ss.clear();
	ss.str(s);

	ss >> format(b, EigenFormat::eigen());
	BOOST_REQUIRE_EQUAL(a.rows(),b.rows());
	BOOST_REQUIRE_EQUAL(a.cols(),b.cols());
	BOOST_CHECK_EQUAL(a,b);

	// clean format
	s = " [ 1, 2, 3, 4] [5, 6,  7 , 8 ]\n[9 ,10 ,  11, 12] \n [  13, 14, 15, 16]";
	ss.clear();
	ss.str(s);

	ss >> format(b, EigenFormat::clean());
	BOOST_REQUIRE_EQUAL(a.rows(),b.rows());
	BOOST_REQUIRE_EQUAL(a.cols(),b.cols());
	BOOST_CHECK_EQUAL(a,b);

	//numpy format
	s = " [[ 1, 2, 3, 4],\n[5, 6,  7 , 8 ],\n[9 ,10 ,  11, 12],\n[  13, 14, 15, 16]]";
	ss.clear();
	ss.str(s);

	ss >> format(b, EigenFormat::python());
	BOOST_REQUIRE_EQUAL(a.rows(),b.rows());
	BOOST_REQUIRE_EQUAL(a.cols(),b.cols());
	BOOST_CHECK_EQUAL(a,b);


	// matlab format
	s = " [] ";
	ss.clear();
	ss.str(s);

	ss >> format(b, EigenFormat::matlab());
	BOOST_REQUIRE_EQUAL(0,b.rows());
	BOOST_REQUIRE_EQUAL(0,b.cols());

	// clean format
	s = "[]";
	ss.clear();
	ss.str(s);

	ss >> format(b, EigenFormat::clean());
	BOOST_REQUIRE_EQUAL(1,b.rows());
	BOOST_REQUIRE_EQUAL(0,b.cols());

	// numpy format
	s = " [  ] ";
	ss.clear();
	ss.str(s);

	ss >> format(b, EigenFormat::python());
	BOOST_REQUIRE_EQUAL(0,b.rows());
	BOOST_REQUIRE_EQUAL(0,b.cols());
}

BOOST_AUTO_TEST_CASE( EigenFormatNaNTest )
{
	Matrix4f a,b;
	a << std::numeric_limits<float>::quiet_NaN(),2,3,4,
		 5,6,7,8,
		 9,10,11,12,
		 13,14,15,16;

	//string s = "   [ 1,2,3,4;5, 6,  7,8 ;  9,10   ,11,12 ;13,14 , 15,16 ]";
	stringstream ss;

	// parse !
	ss << format(a);

	std::cout << "Matrix with nan: " << std::endl << ss.str() << std::endl;

	ss >> format(b);

}

/*
BOOST_AUTO_TEST_CASE( ReadingIntoConstMatrixCompileErrorTest )
{
	MatrixXf b;
	const MatrixXf& constRef = b;


	string s = "   [ 1,2,3,4;5, 6,  7,8 ;  9,10   ,11,12 ;13,14 , 15,16 ]";
	stringstream ss(s);
	ss >> format(constRef);

}
*/
