/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SQLiteTest.C
 *    Test cases for SQLiteDB.
 *
 * @author Tim Langner
 * @date   25.10.2010
 */

#include <boost/test/unit_test.hpp>

#include <database/SQLiteDB.h>
#include <serialization/XMLSerializer.h>


using namespace mira;

typedef boost::shared_ptr<SQLiteDB> SQLiteDBPtr;

SQLiteDBPtr createDB()
{
	remove("test.db");
	SQLiteDBPtr db(new SQLiteDB("test.db"));
	db->exec("create table employee(number int, name char(20));");
	db->exec("insert into employee values (1, 'Homer Simpson');");
	db->exec("insert into employee values (2, 'Patrick Star');");
	db->exec("insert into employee values (3, null);");
	return db;
}

BOOST_AUTO_TEST_CASE(dbCreate)
{
	remove("test.db");
	SQLiteDB db("test.db");
	db.exec("create table employee(number int, name char(20));");
	BOOST_CHECK_EQUAL(db.tableExists("employee"), true);
	BOOST_CHECK_EQUAL(db.tableExists("nonexisting"), false);
	BOOST_CHECK_THROW(db.exec("do something wrong"), mira::XSQLite);
	BOOST_CHECK_THROW(db.query("select invalid syntax"), mira::XSQLite);
	int rows = 0;
	rows += db.exec("insert into employee values (1, 'Homer Simpson');");
	rows += db.exec("insert into employee values (2, 'Patrick Star');");
	BOOST_CHECK_EQUAL(rows, 2);
	rows = db.query("select count(*) from employee;").getValue<int>(0);
	BOOST_CHECK_EQUAL(rows, 2);
}

BOOST_AUTO_TEST_CASE(basicQuery)
{
	SQLiteDBPtr db = createDB();
	SQLiteQuery q = db->query("select * from employee order by 1;");
	BOOST_CHECK_EQUAL(q.getColumnCount(), 2);
	BOOST_CHECK(q.getName(0) == "number");
	BOOST_CHECK(q.getName(1) == "name");
	BOOST_CHECK_THROW(q.getName(2), XAccessViolation);
	BOOST_CHECK_THROW(q.getDataType(2), XAccessViolation);

	BOOST_CHECK_EQUAL(q.getValue<int>(0), 1);
	BOOST_CHECK(q.getValue<std::string>(1) == "Homer Simpson");
	++q;
	BOOST_CHECK_EQUAL(q.getValue<int>(0), 2);
	BOOST_CHECK(q.getValue<std::string>(1) == "Patrick Star");
	BOOST_CHECK(q.isNull(1) == false);
	++q;
	BOOST_CHECK(q.isNull(1) == true);
	BOOST_CHECK_THROW(q.getValue<std::string>(1), XBadCast);
	BOOST_CHECK(q.getValue<std::string>(1, "default") == "default");
}

BOOST_AUTO_TEST_CASE(cachedQuery)
{
	SQLiteDBPtr db = createDB();
	SQLiteQuery bq = db->query("select * from employee order by 1;");
	SQLiteCachedQuery q(bq);
	BOOST_CHECK_EQUAL(q.getColumnCount(), 2);
	BOOST_CHECK(q.getName(0) == "number");
	BOOST_CHECK(q.getName(1) == "name");
	BOOST_CHECK_THROW(q.getName(2), XAccessViolation);
	BOOST_CHECK_THROW(q.getDataType(2), XAccessViolation);

	BOOST_CHECK_EQUAL(q.getValue<int>(0), 1);
	BOOST_CHECK(q.getValue<std::string>(1) == "Homer Simpson");
	++q;
	BOOST_CHECK_EQUAL(q.getValue<int>(0), 2);
	BOOST_CHECK(q.getValue<std::string>(1) == "Patrick Star");
	BOOST_CHECK(q.isNull(1) == false);
	++q;
	BOOST_CHECK(q.isNull(1) == true);
	BOOST_CHECK_THROW(q.getValue<std::string>(1), XBadCast);
	BOOST_CHECK(q.getValue<std::string>(1, "default") == "default");
}

BOOST_AUTO_TEST_CASE(dbReflect)
{
	SQLiteDBPtr db = createDB();
	SQLiteQuery bq = db->query("select * from employee order by 1;");
	SQLiteCachedQuery q(bq);

	XMLDom xml;
	XMLSerializer xs(xml);
	xs.serialize("Query", q);

	SQLiteCachedQuery cq;
	XMLDeserializer ds(xml);
	ds.deserialize("Query", cq);

	BOOST_CHECK_EQUAL(cq.getColumnCount(), 2);
	BOOST_CHECK(cq.getName(0) == "number");
	BOOST_CHECK(cq.getName(1) == "name");
	BOOST_CHECK_THROW(cq.getName(2), XAccessViolation);
	BOOST_CHECK_THROW(cq.getDataType(2), XAccessViolation);

	BOOST_CHECK_EQUAL(cq.getValue<int>(0), 1);
	BOOST_CHECK(cq.getValue<std::string>(1) == "Homer Simpson");
	++cq;
	BOOST_CHECK_EQUAL(cq.getValue<int>(0), 2);
	BOOST_CHECK(cq.getValue<std::string>(1) == "Patrick Star");
	BOOST_CHECK(cq.isNull(1) == false);
	++cq;
	BOOST_CHECK(cq.isNull(1) == true);
	BOOST_CHECK_THROW(cq.getValue<std::string>(1), XBadCast);
	BOOST_CHECK(cq.getValue<std::string>(1, "default") == "default");
}

BOOST_AUTO_TEST_CASE(CopyMoveQueryTest)
{
	SQLiteDBPtr db = createDB();

	SQLiteQuery bq;
	BOOST_CHECK(!bq.valid());

	bq = db->query("select * from employee order by 1;"); // uses move, not copy
	BOOST_CHECK(bq.valid());

	SQLiteQuery bq2;
//	bq2 = bq; // this must not compile
	bq2 = std::move(bq);
	BOOST_CHECK(!bq.valid());
	BOOST_CHECK(bq2.valid());
}
