/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Print.C
 *    Provides string generation out of JSON values for the print() method.
 *
 * @author Erik Einhorn
 * @date   2011/10/15
 */

#include <serialization/Print.h>

using namespace json_spirit;

namespace mira { namespace Private {

///////////////////////////////////////////////////////////////////////////////

// large parts of the following code were taken from json/json_spirit_writer_template.h
class JSONPrettyPrintGenerator
{
	typedef JSONValue    ValueType;
	typedef std::ostream OstreamType;

	typedef ValueType::Config_type  ConfigType;
	typedef ConfigType::String_type StringType;
	typedef ConfigType::Object_type ObjectType;
	typedef ConfigType::Array_type  ArrayType;
	typedef StringType::value_type  CharType;
	typedef ObjectType::value_type  ObjectMemberType;

public:

	JSONPrettyPrintGenerator(const ValueType& value, OstreamType& os, int precision) :
		mStream(os), mIndent(0), mAdditionalIndent(0), mPrecision(precision)
	{
		output(value);
	}

private:

	void output(const ValueType& value)
	{
		switch(value.type())
		{
		case obj_type:   output(value.get_obj());   break;
		case array_type: output(value.get_array()); break;
		case str_type:   output(value.get_str());   break;
		case bool_type:  output(value.get_bool());  break;
		case int_type:   output_int(value);         break;
		case real_type:  mStream << std::showpoint
		                         << std::setprecision(mPrecision < 0 ?
		                                              json::JSONDefaultPrecision::get() :
		                                              mPrecision)
		                         << value.get_real();     break;
		case null_type:  mStream << "null";               break;
		default: assert(false); break;
		}
	}

	void output(const ObjectType& obj) {
		outputArrayOrObject(obj, '{', '}');
	}

	void output(const ArrayType& arr) {
		outputArrayOrObject(arr, '[', ']');
	}

	void output(const ObjectMemberType& member) {
		StringType name = ConfigType::get_name(member);
		output(name);
		mStream << " : ";
		mAdditionalIndent = name.length() + 2 + 3;
		output(ConfigType::get_value(member));
		mAdditionalIndent = 0;
	}

	void output_int(const ValueType& value)
	{
		if(value.is_uint64())
			mStream << value.get_uint64();
		else
			mStream << value.get_int64();
	}

	void output(const StringType& s)
	{
		typedef StringType::const_iterator IterType;

		mStream << '\"';
		++mAdditionalIndent;

		StringType line;
		bool firstLine=true;
		const IterType end(s.end());
		for(IterType i = s.begin(); i != end; ++i)
		{
			const CharType c(*i);
			switch(c)
			{
				case '\n':
					if(!firstLine) {
						new_line();
						indentFull();
					}
					mStream << line;
					line.clear();
					firstLine=false;
					break;
				case '\r':
					// eat carriage return
					break;
				default:
					line += c;
					break;
			}
		}
		if(!line.empty()) {
			if(!firstLine) {
				new_line();
				indentFull();
			}
			mStream << line;
		}

		mStream << '\"';
	}

	void output(bool b)
	{
		if(b)
			mStream << "true";
		else
			mStream << "false";
	}

	template< class T >
	void outputArrayOrObject(const T& t, CharType start_char,
	                         CharType end_char)
	{
		mStream << start_char; new_line();
		++mIndent;

		for(typename T::const_iterator i = t.begin(); i != t.end(); ++i)
		{
			indent(); output(*i);
			typename T::const_iterator next = i;
			if(++next != t.end())
				mStream << ',';
			new_line();
		}
		--mIndent;
		indent(); mStream << end_char;
	}

	void indent()
	{
		for(int i=0; i<mIndent; ++i)
			mStream << "    ";
	}

	void indentFull()
	{
		indent();
		for(int i=0; i<mAdditionalIndent; ++i)
			mStream << " ";
	}

	void space() {
		mStream << ' ';
	}

	void new_line()	{
		mStream << '\n';
	}

	OstreamType& mStream;
	int mIndent;
	int mAdditionalIndent;
	int mPrecision;
};

void jsonPrettyPrint(const JSONValue& value, std::ostream& os, PrintFormat format, int precision)
{
	switch(format) {
		case COMPACT_PRINT   : json::write(value, os, false, precision); break;
		case FORMATTED_PRINT : json::write(value, os, true, precision); break;
		case PRETTY_PRINT    : JSONPrettyPrintGenerator(value, os, precision); break;
		default: break;
	}
}

///////////////////////////////////////////////////////////////////////////////

} } // namespace

