/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MD5.C
 *    Implementation of OpenSSLWrapperMD.h
 *
 * @author Christian Martin
 * @date   2010/09/09
 */

#include <security/OpenSSLWrapperMD.h>

#include <openssl/evp.h>

#include <error/Exceptions.h>

#include "../OpenSSLHelper.h"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

struct OpenSSLWrapperMD::Context
{
	std::string algorithm;
	EVP_MD_CTX* evp;
};

///////////////////////////////////////////////////////////////////////////////

OpenSSLWrapperMD::OpenSSLWrapperMD(const std::string& algo) :
	mCtx(NULL)
{
	OpenSSLCleanup::instance();

	mCtx = new Context;
	mCtx->algorithm = algo;
	mCtx->evp = EVP_MD_CTX_new();
	if (mCtx->evp == NULL)
		MIRA_THROW(XSystemCall, "EVP_MD_CTX_new failed.");

	reset();
}

OpenSSLWrapperMD::~OpenSSLWrapperMD()
{
	if (mCtx->evp)
		EVP_MD_CTX_free(mCtx->evp);
	delete mCtx;
	mCtx = NULL;
}

void OpenSSLWrapperMD::reset()
{
	const EVP_MD* md = EVP_get_digestbyname(mCtx->algorithm.c_str());
	if (md == NULL)
		MIRA_THROW(XSystemCall, "Unknown message digest algorithm '"
			<< mCtx->algorithm << "'");

	if (!EVP_DigestInit_ex2(mCtx->evp, md, NULL))
		MIRA_THROW(XSystemCall, "EVP_DigestInit_ex2 failed.");
}

HashDigest OpenSSLWrapperMD::getDigest() const
{
	uint8 digest[EVP_MAX_MD_SIZE];
	unsigned int dlen = 0;

	if (!EVP_DigestFinal_ex(mCtx->evp, digest, &dlen))
		MIRA_THROW(XSystemCall, "EVP_DigestFinal_ex failed.");

	HashDigest res;
	res.insert(res.begin(), &digest[0], &digest[dlen]);

	return res;
}

std::streamsize OpenSSLWrapperMD::xsputn(const char* s, std::streamsize n)
{
	if (!EVP_DigestUpdate(mCtx->evp, s, n))
		MIRA_THROW(XSystemCall, "EVP_DigestUpdate failed.");

	return(n);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace
