/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Typename.C
 *    Implementation of Typename.h.
 *
 * @author Tim Langner
 * @date   2010/10/19
 */

#include <unordered_map>

#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/erase.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/find.hpp>
#include <boost/regex.hpp>

#include <error/Exceptions.h>
#include <platform/Typename.h>
#include <platform/Platform.h>

using namespace boost::algorithm;


#ifdef MIRA_LINUX

#include <cxxabi.h>

namespace mira { namespace Private {

///////////////////////////////////////////////////////////////////////////////

Typename demangle(const std::type_info& t)
{
	// costly calculation and result should not depend on anything but input
	// --> cache it!
	static std::unordered_map<Typename, Typename> nameMap;

	Typename raw = t.name();

	std::unordered_map<Typename, Typename>::const_iterator it = nameMap.find(raw);
	if (it != nameMap.cend())
		return it->second;

	Typename result;
	int status;
	char* realname;
	realname = abi::__cxa_demangle(t.name(), 0, 0, &status);
	if ( realname == NULL )
		MIRA_THROW(XSystemCall, 
		          "Demangling of typename '" << t.name() << "' failed");
	result = Typename(realname);
	free(realname);

	// the new cxx11 abi adds another namespace to common classes like string and list
	replace_all(result,
	            "std::__cxx11::",
	            "std::");
	// somehow the cxx11 abi mangles std::string differently
	replace_all(result,
	            "std::basic_string<char, std::char_traits<char>,"
	            " std::allocator<char> >",
	            "std::string");

	// remove const before (
	replace_all(result, "const (", "(");
	// remove all white spaces before and after parenthesis, * and colons
	replace_all(result, " *", "*");
	replace_all(result, "* ", "*");
	replace_all(result, ", ", ",");
	replace_all(result, " ,", ",");
	replace_all(result, "> ", ">");
	replace_all(result, " >", ">");
	replace_all(result, "< ", "<");
	replace_all(result, " <", "<");
	replace_all(result, ") ", ")");
	replace_all(result, " )", ")");
	replace_all(result, "( ", "(");
	replace_all(result, " (", "(");
	// replace 64bit data types
	replace_all(result, "long double", "double128");
	replace_all(result, "long long", "int64");
	boost::regex re("\\blong\\b");
#ifdef MIRA_ARCH64
	std::string format("int64");
#else
	std::string format("int");
#endif
	result = boost::regex_replace(result, re, format);

	nameMap[raw] = result;
	return result;
}

}}

#else

namespace mira { namespace Private {

/// normal anonymous namespaces get this prefix
#define MIRA_ANONYMOUS_NAMESPACE "`anonymous namespace'::"
/// anonymous namespaces used as template parameters get this bizarre prefix
#define MIRA_ANONYMOUS_TPL_NAMESPACE "A0xe57121d0::"

std::string::iterator match(std::string::iterator i, std::string::iterator end,
                            char openChar, char closeChar)
{
	int count = 1;
	while (count)
	{
		++i;
		if (*i == openChar)
			++count;
		else
			if (*i == closeChar)
				--count;
	}
	return i;
}

Typename demangle(const std::type_info& t)
{
	// costly calculation and result should not depend on anything but input
	// --> cache it!
	static std::map<std::type_index, Typename> nameMap;

	std::type_index index(t);

	std::map<std::type_index, Typename>::const_iterator it = nameMap.find(index);
	if (it != nameMap.cend())
		return it->second;

	Typename result = t.name();
	/*
	 * std::string specifically gets special handling when mangled by GCC.
	 * Usually typedefs get mangled according to their actual type.
	 */
	replace_all(result,
	            "std::basic_string<char,struct std::char_traits<char>,"
	            "class std::allocator<char> >",
	            "std::string");
	// remove struct, class and calling conventions
	erase_all(result, "struct ");
	erase_all(result, "class ");
	erase_all(result, "enum ");
	erase_all(result, "__cdecl");
	erase_all(result, "__thiscall ");
	// MSVC x64 adds " __ptr64" qualifier to pointers
	erase_all(result, " __ptr64");
	// replace 64bit data types
	replace_all(result, "long double", "double128");
	replace_all(result, "__int64", "int64");
	replace_all(result, "long long", "int64");
	boost::regex re("\\blong\\b");
#ifdef MIRA_ARCH64
	std::string format("int64");
#else
	std::string format("int");
#endif
	result = boost::regex_replace(result, re, format);

	// remove old void on functions without parameters
	replace_all(result, "(void)", "()");

	// remove all white spaces before and after parenthesis, * and colons
	replace_all(result, " *", "*");
	replace_all(result, "* ", "*");
	replace_all(result, ", ", ",");
	replace_all(result, " ,", ",");
	replace_all(result, "> ", ">");
	replace_all(result, " >", ">");
	replace_all(result, "< ", "<");
	replace_all(result, " <", "<");
	replace_all(result, "( ", "(");
	replace_all(result, " (", "(");

	// Some anonymous namespace handling
	replace_all(result, MIRA_ANONYMOUS_NAMESPACE, "(anonymous namespace)::");
	replace_all(result, MIRA_ANONYMOUS_TPL_NAMESPACE, "(anonymous namespace)::");

	// Windows compiler adds some consts before, after and between parenthesis - remove them
	while (boost::iterator_range<std::string::iterator> ir = find_first(result, "const "))
	{
		std::string spaceconst = " const";
		bool done = false;
		for (std::string::iterator i = ir.end(); i != result.end() && !done; ++i)
		{
			std::string thischar(1, *i);
			if (find_first(">&*", thischar))
			{
				result.insert(i, spaceconst.begin(), spaceconst.end());
				done = true;
			}
			else
				if (find_first("()", thischar))
					done = true;
				else
					if (*i == '<')
						i = match(i, result.end(), '<', '>');
					else
						if (*i == ',')
						{
							// are we within <>?
							for (std::string::iterator t = i; t != result.begin() && !done; --t)
								if (*t == '(')
								{
									std::string::iterator e = match(t, result.end(), '(', ')');
									if (e > i) // i within () but not <>
										done = true;
								}
								else
									if (*t == '<')
									{
										std::string::iterator e = match(t, result.end(), '<', '>');
										if (e > i)
										{ // i within <>!
											result.insert(i, spaceconst.begin(),
													spaceconst.end());
											done = true;
										}
									}
						}
		}
		replace_first(result, "const ", "");
	}

	// remove all double white spaces
	replace_all(result, "  ", " ");

	nameMap[index] = result;
	return result;
}

///////////////////////////////////////////////////////////////////////////////

}}

#endif
