/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VacantClass.C
 *    $Implementation of VacantClass.h$.
 *
 * @author Ronny Stricker
 * @date   2010/11/03
 */

#include <factory/VacantClass.h>

#include <map>
#include <cstdarg>

#include <boost/shared_ptr.hpp>

#include <error/Exceptions.h>
#include <factory/Class.h>
#include <factory/Factory.h>

namespace mira {

#define MIRA_LOAD_LIB_ONCE                                                     \
	/* save me from being deleted during function call */                      \
	boost::shared_ptr<Class> tSavePointer = ClassFactory::instance().mClasses[ getIdentifier() ]; \
	/* don't try to load the library a second time */                          \
	/* multiple loading would not make it better cause we will run into a */   \
	/* infinity loop */                                                        \
	if ( mLoadInitiated ) {                                                    \
		MIRA_THROW(XLogical, "Object " + getIdentifier() +" not registered"    \
		           " in lib " + mLib+ "! (did you forget the factory macro?)");\
	}                                                                          \
	/* try to replace myself with real class :-(   */                          \
	getLibraryLoader().loadLibrary( mLib );                                    \
	mLoadInitiated = true;

//////////////////////////////////////////////////////////////////////////////

// *****************************************************************************
// *** VacantClass implementation
// *****************************************************************************

int VacantClass::getTypeId() const
{
	MIRA_LOAD_LIB_ONCE
	return ClassFactory::getClassByIdentifier( getIdentifier() ).getTypeId();
}

Typename VacantClass::getTypename() const
{
	MIRA_LOAD_LIB_ONCE
	return ClassFactory::getClassByIdentifier( getIdentifier() ).getTypename();
}


Object* VacantClass::newInstance() const
{
	MIRA_LOAD_LIB_ONCE
	return ClassFactory::newInstance<Object>( getIdentifier() );
}

Object* VacantClass::newInstance( int paramCount, ... ) const
{
	MIRA_LOAD_LIB_ONCE
	std::va_list ap;
	va_start(ap, paramCount);
	return ClassFactory::newInstance<Object>( getIdentifier(), paramCount, ap );
}

Object* VacantClass::newInstance( std::string const& childIdentifier ) const
{
	MIRA_LOAD_LIB_ONCE
	return ClassFactory::newInstance<Object>( childIdentifier );
}

Object* VacantClass::newInstance( std::string const& childIdentifier,
                                 int paramCount, ... ) const
{
	MIRA_LOAD_LIB_ONCE
	std::va_list ap;
	va_start(ap, paramCount);
	return ClassFactory::newInstance<Object>(
				childIdentifier, paramCount, ap );
}

Object* VacantClass::newVAInstance( int paramCount, std::va_list list ) const
{
	MIRA_LOAD_LIB_ONCE
	return ClassFactory::newInstance<Object>(
				getIdentifier(), paramCount, list );
}

SharedLibraryLoader& VacantClass::getLibraryLoader()
{
	static SharedLibraryLoader sLibraryLoader;
	return sLibraryLoader;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
