/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SymbolTable.h
 *    Provides class for obtaining information like file name and line
 *    number of a certain code portion from the symbol table. (if debug
 *    information are compiled in)
 *
 * @author Erik Einhorn, Christian Martin
 * @date   2009/12/26
 */
#ifndef _MIRA_SYMBOLTABLE_H_
#define _MIRA_SYMBOLTABLE_H_

#include <iostream>
#include <string>
#include <map>

#include <platform/Types.h>
#include <utils/Singleton.h>

#ifndef MIRA_WINDOWS
# error "This file is only for Windows!"
#endif

#include <Tlhelp32.h>
#include <DbgHelp.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @brief A class for accessing the symbol table.
 *
 * This class obtains informations like file name and line number of the
 * underlying code from the symbol table. It is only working if debug symbols
 * are compiled in.
 *
 */
class SymbolTable : public LazySingleton<SymbolTable>
{
public:
	/**
	 * Inner class representing a location in a source file (line number, filename)
	 */
	class SourceLocation
	{
	public:
		SourceLocation() : mLine(0) {}
		SourceLocation(const std::string& filename, int line) :
			mFilename(filename), mLine(line) {}

		/// return the filename of the source location
		const std::string& filename() const { return mFilename; }

		/// returns the line of the location
		int line() const { return mLine; }

		friend std::ostream& operator<< (std::ostream& os, const SourceLocation& loc) {
			if(!loc.filename().empty())
				os << loc.filename() << ":" << loc.line();
			else
				os << "???";
			return os;
		}

	private:
		std::string mFilename;
		int mLine;
	};
	
	//////////////////////////////////////////////////////////////////////////

public:
	SymbolTable();

public:
	/// Destructor
	~SymbolTable();

	//////////////////////////////////////////////////////////////////////////

public:
	/**
	 * Converts the address given to its corresponding source location
	 */
	SourceLocation sourceLocation(uint64 addr) const;

	/**
	 * Get the module name for the address. Returns @c true on success.
	 */
	bool getModuleNameFromAddr(DWORD64 iAddress, std::string& oName) const;

	/*
	 * Get the symbol name for the address. Returns @c true on success.
	 */
	bool getSymbolNameFromAddr(DWORD64 iAddress, std::string& oName, int64& oOffset) const;

	/*
	 * Get file name and line number for the address. Returns @c true on success.
	 */
	bool getFileAndLineFromAddr(DWORD64 iAddress, uint32& oLine, std::string& oFileName) const;

	//////////////////////////////////////////////////////////////////////////

private:
	BOOL LoadModule(LPCSTR iBinaryName, LPCSTR iModuleName, 
	                DWORD64 iDLLBaseAddr, DWORD iDLLSize);

	//////////////////////////////////////////////////////////////////////////

private:
	bool mInitialized;
	HANDLE mProcess;
	DWORD mOldSymOpts;

	std::map<DWORD64, std::string> mModuleNames;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
