/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ToString.h
 *    Contains toString and fromString functions for converting
 *    data types to strings and the other way round.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/07/08
 */

#ifndef _MIRA_TOSTRING_H_
#define _MIRA_TOSTRING_H_

#include <iomanip>

#ifndef Q_MOC_RUN
#include <boost/numeric/conversion/cast.hpp>
#include <boost/lexical_cast.hpp>
#endif

#include <error/Exceptions.h>

#include <math/Math.h>

///////////////////////////////////////////////////////////////////////////////

#define MIRA_TOSTRING_NAN          "nan"
#define MIRA_TOSTRING_POSITIVE_INF "inf"
#define MIRA_TOSTRING_NEGATIVE_INF "-inf"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace Private {

/// Generic conversion function
template <typename T>
inline std::string genericToString(const T& value, int precision)
{
	std::stringstream ss;
	if(precision>=0)
		ss << std::setprecision(precision);
	ss << value;
	if(ss.fail())
		MIRA_THROW(XIO, "Failed to convert value into a string");
	return ss.str();
}

/// Generic conversion function
template <typename T>
inline T genericFromString(const std::string& str)
{
	T value;
	std::stringstream ss(str);
	ss >> value;
	if(ss.fail()) // make sure that we have no error flag set
		MIRA_THROW(XIO, "Failed to convert value from string '" << str << "'");

	return value;
}

/// Helper for floating point numbers
template <typename T>
inline std::string fpToString(const T& value, int precision)
{
	if(boost::math::isnan(value))
		return MIRA_TOSTRING_NAN;
	else if(boost::math::isinf(value)) {
		if(value>0)
			return MIRA_TOSTRING_POSITIVE_INF;
		else
			return MIRA_TOSTRING_NEGATIVE_INF;
	}
	else
		return genericToString<T>(value, precision);
}

/// Helper for floating point numbers
template <typename T>
inline T fpFromString(const std::string& str)
{
	try {
		return genericFromString<T>(str);
	} catch(XIO&) {
		if(str==MIRA_TOSTRING_NAN)
			return std::numeric_limits<T>::quiet_NaN();
		else if(str==MIRA_TOSTRING_POSITIVE_INF)
			return std::numeric_limits<T>::infinity();
		else if(str==MIRA_TOSTRING_NEGATIVE_INF)
			return -std::numeric_limits<T>::infinity();
		else
			throw; // we could not handle the exception, so rethrow it
	}
}

/// Helper for char, signed char, etc
template <typename T>
inline std::string charToString(const T& value)
{
	return genericToString<int>((int)value, 0);
}

/// Helper for char, signed char, etc
template <typename T>
inline T charFromString(const std::string& str)
{
	int val = genericFromString<int>(str);

	if(val>std::numeric_limits<T>::max())
		MIRA_THROW(XIO, "Overflow while converting value from string '" << 
		           str << "', max: " << (int)std::numeric_limits<T>::max());
	if(val<std::numeric_limits<T>::min())
		MIRA_THROW(XIO, "Underflow while converting value from string '" << 
		           str << "', min: " << (int)std::numeric_limits<T>::min());

	return static_cast<T>(val);
}

} // Private
///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * Can be used with fromString to convert hex strings into numbers.
 * This is necessary since std streams and boost lexical_cast can not deal with
 * hex strings.
 *
 * \code
 * uint32 i = fromString<FromHex<uint32>>("0x010203");
 * \endcode
 */
template <typename T>
class FromHex
{
public:
	operator T() const { return value; }
	friend std::istream& operator>>(std::istream& iStream, FromHex<T>& oValue)
	{
		iStream >> std::hex >> oValue.value;
		return iStream;
	}

	T value;
};

/**
 * Can be used with toString to convert values to their string hex representation.
 * \code
 * std::string s = toString<ToHex<uint32>>(255);
 * \endcode
 */
template <typename T>
class ToHex
{
public:
	ToHex(T v) : value(v) {}
	operator T() const { return value; }
	friend std::ostream& operator<<(std::ostream& oStream, const ToHex<T>& iValue)
	{
		oStream << std::hex << iValue.value;
		return oStream;
	}

	T value;
};

/**
 * Can be used with fromString to convert oct strings into numbers.
 * \code
 * uint32 i = fromString<FromOct<uint32>>("644");
 * \endcode
 */
template <typename T>
class FromOct
{
public:
	operator T() const { return value; }
	friend std::istream& operator>>(std::istream& iStream, FromOct<T>& oValue)
	{
		iStream >> std::oct >> oValue.value;
		return iStream;
	}

	T value;
};

/**
 * Can be used with toString to convert values to their string oct representation.
 * \code
 * std::string s = toString<ToOct<uint32>>(255);
 * \endcode
 */
template <typename T>
class ToOct
{
public:
	ToOct(T v) : value(v) {}
	operator T() const { return value; }
	friend std::ostream& operator<<(std::ostream& oStream, const ToOct<T>& iValue)
	{
		oStream << std::oct << iValue.value;
		return oStream;
	}

	T value;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Converts any data type to string (the data type must support the stream << operator).
 * Supports precision parameter.
 * @throw XIO when conversion fails.
 */
template <typename T>
std::string toString(const T& value, int precision=-1)
{
	return Private::genericToString<T>(value, precision);
}

/**
 * Converts a string to any data type that supports the stream >> operator.
 * @throw XIO when conversion failed.
 * @note On some systems an XIO exception is thrown when converting an negative
 *       value to an unsigned data type e.g. fromString<uint16>("-1"). On some
 *       systems this exception is not thrown and the value may overflow.
 */
template <typename T>
T fromString(const std::string& str)
{
	return Private::genericFromString<T>(str);
}


/// Specialization for char.
template<>
inline std::string toString<char>(const char& value, int precision)
{
	return Private::charToString<char>(value);
}

/// Specialization for char.
template <>
inline char fromString<char>(const std::string& str)
{
	return Private::charFromString<char>(str);
}

/// Specialization for signed char.
template<>
inline std::string toString<signed char>(const signed char& value, 
                                         int precision)
{
	return Private::charToString<signed char>(value);
}

/// Specialization for signed char.
template <>
inline signed char fromString<signed char>(const std::string& str)
{
	return Private::charFromString<signed char>(str);
}

/// Specialization for unsigned char.
template<>
inline std::string toString<unsigned char>(const unsigned char& value, 
                                           int precision)
{
	return Private::charToString<unsigned char>(value);
}

/// Specialization for unsigned char.
template <>
inline unsigned char fromString<unsigned char>(const std::string& str)
{
	return Private::charFromString<unsigned char>(str);
}


/// Specialization for string
template<>
inline std::string toString<std::string>(const std::string& value, 
                                         int precision)
{
	return value;
}

/// Specialization for string
template <>
inline std::string fromString<std::string>(const std::string& str)
{
	return str;
}

/// Specialization for float
template<>
inline std::string toString<float>(const float& value, int precision)
{
	return Private::fpToString<float>(value, precision);
}

/// Specialization for float
template <>
inline float fromString<float>(const std::string& str)
{
	return Private::fpFromString<float>(str);
}

/// Specialization for double
template<>
inline std::string toString<double>(const double& value, int precision)
{
	return Private::fpToString<double>(value, precision);
}

/// Specialization for double
template <>
inline double fromString<double>(const std::string& str)
{
	return Private::fpFromString<double>(str);
}

/// Specialization for long double
template<>
inline std::string toString<long double>(const long double& value, 
                                         int precision)
{
	return Private::fpToString<long double>(value, precision);
}

/// Specialization for long double
template <>
inline long double fromString<long double>(const std::string& str)
{
	return Private::fpFromString<long double>(str);
}

/// Specialization for bool.
template<>
inline std::string toString<bool>(const bool& value, int precision)
{
	if ( value )
		return "true";
	else
		return "false";
}

/// Specialization for bool.
template <>
inline bool fromString<bool>(const std::string & str)
{
	return str == "true";
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
