/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PrintProgressBar.h
 *    Methods for displaying a progress bar in the console
 *
 * @author Erik Einhorn
 * @date   2013/11/20
 */

#ifndef _MIRA_PRINTPROGRESSBAR_H_
#define _MIRA_PRINTPROGRESSBAR_H_

#include <iostream>

#include <utils/Time.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Prints the header part of a progress bar.
 * See printProgressBar() for more information.
 */
inline void printProgressBarHeader()
{
	std::cout << "0%   10   20   30   40   50   60   70   80   90   100%" << std::endl;
	std::cout << "|----|----|----|----|----|----|----|----|----|----|" << std::endl;
}

/**
 * Prints a progress bar in the console and shows the approximated time
 * until the progress is finished.
 *
 * @param value   The current progress value
 * @param maximum The maximum value that corresponds to 100%
 * @param startTime The time stamp of the start of the whole operation (needed to estimate remaining time)
 * @param updateInterval Optional parameter that controls how often the progress bar is updated
 *
 * Example:
 * \code
 * printProgressBarHeader();
 * Time start=Time::now();
 * for(int i=0; i<1000; ++i)
 * {
 *     printProgressBar(i,1000,start, 100);
 *     ... // do something
 * }
 * printFinishedProgressBar();
 * \endcode
 */
inline void printProgressBar(int value, int maximum, const Time& startTime, int updateInterval=1)
{
	if ((value%updateInterval) != 0)
		return;

	std::cout << "\r";
	int ticks = (int64)value * 50 / maximum;
	for(int i=0; i<ticks; ++i)
		std::cout << "*";

	for(int i=ticks; i<50; ++i)
		std::cout << " ";

	if ( value > 0)
		std::cout << "   finished in approx. "
		          << Duration::milliseconds((maximum-value)*(Time::now()-startTime).totalMilliseconds()/value);
	std::flush(std::cout);
}

/// @see printProgressBar()
inline void printFinishedProgressBar()
{
	std::cout << "\r";
	for(int i=0; i<=50; ++i)
		std::cout << "*";
	std::cout << "  finished                           " << std::endl;
	std::flush(std::cout);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
