/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file HasMember.h
 *    Macros for checking the existence of class members.
 * @author Erik Einhorn
 * @date   2010/06/30
 */

#ifndef MIRA_HASMEMBER_H_
#define MIRA_HASMEMBER_H_

#ifndef Q_MOC_RUN
#include <boost/type_traits/integral_constant.hpp>
#include <boost/type_traits/is_class.hpp>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
struct HasMemberCheckNonClassTypeDummy {};
///@endcond

///////////////////////////////////////////////////////////////////////////////

}

///////////////////////////////////////////////////////////////////////////////
/*
 * The following macro makes usage of SFINAE (Substitution failure is not an error).
 *
 * For more details on member function detection see:
 * http://stackoverflow.com/questions/1966362/sfinae-to-check-for-inherited-member-functions
 *
 * And for an explanation see:
 * http://cplusplus.co.il/2009/09/11/substitution-failure-is-not-an-error-1/
 */

#define MIRA_MEMBER_DETECTOR_EX(Identifier, Member)                             \
template <typename Type>                                                       \
class MIRAHasMemberCheck##Identifier                                            \
{                                                                              \
	struct NotFound { char x[1]; };                                            \
	struct Found    { char x[2]; };                                            \
	struct BaseMixin {                                                         \
		void Member(){}                                                        \
	};                                                                         \
                                                                               \
	struct Base : public Type, public BaseMixin {};                            \
	template <typename T, T t>  class Helper{};                                \
                                                                               \
	template <typename U>                                                      \
	static NotFound deduce(U*, Helper<void (BaseMixin::*)(), &U::Member>* = 0);\
	static Found deduce(...);                                                  \
                                                                               \
public:                                                                        \
	enum { value = (sizeof(Found) == sizeof(deduce((Base*)(0))))};             \
	typedef boost::integral_constant<bool,                                     \
	        (sizeof(Found) == sizeof(deduce((Base*)(0)))) > type;              \
                                                                               \
};


#define MIRA_MEMBER_DETECTOR(Member) MIRA_MEMBER_DETECTOR_EX(Member,Member)

///@cond INTERNAL
template <typename T, typename IsClassType>
struct MIRAHasMemberCheckTraitHelper { typedef T type; };

// use HasMemberCheckNonClassTypeDummy for non class-types to prevent 
// compiler error
template <typename T> 
struct MIRAHasMemberCheckTraitHelper<T, boost::false_type> {
	typedef mira::HasMemberCheckNonClassTypeDummy type;
};

template <typename T>
struct MIRAHasMemberCheckTrait {
	typedef typename MIRAHasMemberCheckTraitHelper<T, 
		typename boost::is_class<T>::type >::type type;
};
///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * Macro checking the existence of a class member.
 * \code
 *     class MyClass {
 *     public:
 *         int i;
 *     }
 *     MIRA_MEMBER_DETECTOR(i)
 *     if (MIRA_HAS_MEMBER(MyClass, i)::value) {
 *        ...
 *     }
 * \endcode
 */
#define MIRA_HAS_MEMBER(Class, Identifier) \
		MIRAHasMemberCheck##Identifier<MIRAHasMemberCheckTrait<Class>::type>

#define MIRA_HAS_MEMBER_TEMPLATE(Class, Identifier) \
		MIRAHasMemberCheck##Identifier<typename MIRAHasMemberCheckTrait<Class>::type>

///////////////////////////////////////////////////////////////////////////////

#endif
