/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CompressBuffer.h
 *    Compression of uint8 Buffers using libzip.
 *
 * @author Tim Langner
 * @date   2011/01/27
 */

#ifndef _MIRA_COMPRESSBUFFER_H_
#define _MIRA_COMPRESSBUFFER_H_

#include <utils/Buffer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Determine the maximum size of the compressed data for uncompressed data 
 * with a given length.
 */
uint32 MIRA_BASE_EXPORT getMaxCompressedLength(uint32 uncompressedSize);

/**
 * Compress a given buffer into a second buffer using libzip.
 * The dest buffer can be resized (e.g. by determining the maximum size of the 
 * compressed data by calling getMaxCompressedLength()) before calling this 
 * method to speed up compression. The buffer will be resized if the compressed 
 * data does not fit in.
 * @param[in] src The data to be compressed
 * @param[in,out] ioDest The compressed data
 * @param[in] compressionLevel ranges from (-1..9)
 *            -1 = default,
 *             0 = no compression,
 *             1 = best speed
 *             9 = best compression
 */
void MIRA_BASE_EXPORT compressBuffer(const Buffer<uint8>& src,
                                     Buffer<uint8>& ioDest,
                                     int compressionLevel = -1);

/**
 * Uncompress a given buffer into a second buffer using libzip.
 * To speed up decompression when the size of the uncompressed buffer is know
 * the dest buffer can be resized to the known size. The buffer will be resized
 * if the uncompressed data does not fit in.
 */
void MIRA_BASE_EXPORT uncompressBuffer(const Buffer<uint8>& src,
                                       Buffer<uint8>& ioDest);

}

///////////////////////////////////////////////////////////////////////////////

#endif
