/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CyclicRunnable.h
 *    Runnable concept that executes an operation repeatedly in an defined interval.
 *
 * @author Tim Langner
 * @date   2010/09/07
 */

#ifndef _MIRA_CYCLICRUNNABLE_H_
#define _MIRA_CYCLICRUNNABLE_H_

#ifndef Q_MOC_RUN
#include <boost/function.hpp>
#endif

#include <thread/ThreadMonitor.h>
#include <utils/Time.h>
#include <utils/ToString.h>

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

/**
 * This class extends threads by the ability to execute an operation repeatedly
 * in an defined interval. By binding the operator() of this class to a boost
 * thread the given functor is executed in the given interval.
 * \code
 * // creates a cyclic runnable, that calls myCyclicFunction every 500 ms
 * CyclicRunnable cyclic(myCyclicFunction, Duration::milliseconds(500));
 *
 * // launch the cyclic runnable in a separate thread
 * boost::thread t(boost::ref(cyclic));
 * // also boost::thread t(boost::bind(&CyclicRunnable::operator(), &cyclic)) can be used.
 * \endcode
 *
 * @note If the operation that is processed repeatedly takes more time than the
 *       specified interval, the interval cannot be hold and the operation is
 *       called as fast as possible.
 *
 * @ingroup ThreadModule
 */
class CyclicRunnable : boost::noncopyable
{
public:
	/**
	 * Constructs a CyclicRunnable with given functor. This uses a default
	 * interval of 1 second
	 * @param[in] fn The function to execute repeatedly.
	 * @param[in] name The optional name of the thread that is registered in the
	 * Threadmanager.
	 */
	CyclicRunnable(boost::function<void ()> fn, const std::string& name="") :
		cycleTime(Duration::seconds(1)),
		mFunction(fn),
		mName(name)
	{
		if ( mName.empty() )
			mName = "CyclicRunnable " + toString(this);
	}

	/**
	 * Constructs a CyclicRunnable with given functor and interval
	 * @param[in] fn The function to execute repeatedly.
	 * @param[in] interval The interval (cycle time) between each execution of fn.
	 * @param[in] name The optional name of the thread that is registered in the
	 * Threadmanager.
	 */
	CyclicRunnable(boost::function<void ()> fn, const Duration& interval,
	               const std::string& name="") :
		cycleTime(interval),
		mFunction(fn),
		mName(name)
	{}

	/**
	 * Bind this to a thread to execute the given functor in the specified
	 * interval.
	 */
	void operator()()
	{
		if(!ThreadMonitor::isDestroyed())
			ThreadMonitor::instance().addThisThread(mName);

		boost::timed_mutex theMutex;
		boost::system_time endtime = boost::get_system_time();
		while ( !boost::this_thread::interruption_requested() )
		{
			// calculate the time when we want to call the functor again
			// (next interval)
			endtime += cycleTime;
			mFunction();
			// scope lock the mutex
			boost::timed_mutex::scoped_lock lock(theMutex);
			// we try to lock the mutex again and block until we reach the next
			// interval time. if the specified time has already passed we go on
			// immediately.
			theMutex.timed_lock(endtime);
		}
		if(!ThreadMonitor::isDestroyed())
			ThreadMonitor::instance().removeThisThread();
	}

public:

	/// The cycle time, this cyclic runnable calls its assigned function.
	Duration cycleTime;

private:
	boost::function<void ()> mFunction;
	std::string mName;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
