/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyManager.h
 *    PropertyManager that stores PropertyNodes.
 *
 * @author Erik Einhorn
 * @date   2010/07/13
 */

#ifndef _MIRA_PROPERTYMANAGER_H_
#define _MIRA_PROPERTYMANAGER_H_

#include <algorithm>

#ifndef Q_MOC_RUN
#include <boost/scoped_ptr.hpp>
#include <boost/algorithm/string/split.hpp>
#endif

#include <error/Exceptions.h>
#include <error/LoggingCore.h>

#include <serialization/adapters/std/list>
#include <serialization/PropertySerializer.h>

#include <platform/Typename.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * The property manager maintains the properties of objects that are added
 * to the manager. A property is a special member of an object that is
 * reflected via the "property" method to indicate that the member can be
 * changed dynamically at runtime.
 * For each object that is added via the addObject() method, all properties
 * are reflected using the PropertySerializer. For each property a PropertyNode
 * is created and stored within the manager. The value of properties of
 * managed objects can be set and obtained using the setProperty() and
 * getProperty() methods. Moreover, the PropertyNode of a property can be
 * obtained directly through the getPropertyNode() methods.
 */
class PropertyManager
{
public:

	PropertyManager() : mRoot(new RootPropertyNode())
	{}

	/**
	 * Add the specified object to the PropertyManager under the specified
	 * name. After calling this method, the properties of the object can be
	 * obtained and modified using the getProperty() and setProperty() methods.
	 * For each property within the objects members, a PropertyNode is created.
	 */
	template <typename T>
	void addObject(const char* name, T& object)	{
		PropertySerializer s;
		s.reflectProperties(mRoot.get(), name, object);
	}

public:

	/**
	 * Modifies the value of the specified property.
	 * The property is addressed via its full name. The following example:
	 * \code
	 * myObject.myMember.value
	 * \endcode
	 * accesses the property member "value" of the member "myMember"
	 * of the object "myObject" that was added via addObject.
	 *
	 * @throws XBadCast If the specified property is not of type T.
	 * @throws XLogical If the property is read-only, or does not exist.
	 */
	template <typename T>
	void setProperty(const std::string& name, const T& value)
	{
		TypedPropertyNode<T>* node = getPropertyNode<T>(name);
		if(node==NULL)
			MIRA_THROW(XLogical, "The property '" << name << "' does not exist");
		node->set(value);
	}

	/**
	 * Obtains the value of the specified property.
	 * The property is addressed via its full name. The following example:
	 * \code
	 * myObject.myMember.value
	 * \endcode
	 * accesses the property member "value" of the member "myMember"
	 * of the object "myObject" that was added via addObject.
	 *
	 * @throws XBadCast If the specified property is not of type T.
	 * @throws XLogical If the property does not exist.
	 */
	template <typename T>
	T getProperty(const std::string& name) const
	{
		const TypedPropertyNode<T>* node = getPropertyNode<T>(name);
		if(node==NULL)
			MIRA_THROW(XLogical, "The property '" << name << "' does not exist");
		return node->get();
	}

public:

	/**
	 * Returns the root node of the managed property tree. Each child node
	 * of the root node corresponds to an object that was added via addObject().
	 */
	const PropertyNode* getRootNode() const {
		return mRoot.get();
	}

	/**
	 * @copydoc getRootNode() const
	 */
	PropertyNode* getRootNode() {
		return mRoot.get();
	}

	/**
	 * Returns the PropertyNode that represents the specified property.
	 * The property is addressed via its full name. The following example:
	 * \code
	 * myObject.myMember.value
	 * \endcode
	 * returns the PropertyNode of the property member "value"
	 * of the member "myMember" of the object "myObject" that was added via addObject.
	 * Returns nullptr if no property with the given name exists.
	 */
	const PropertyNode* getPropertyNode(const std::string& name) const {
		return mRoot->findChildNode(name);
	}

	/**
	 * @copydoc getPropertyNode(const std::string& name) const
	 */
	PropertyNode* getPropertyNode(const std::string& name) {
		return mRoot->findChildNode(name);
	}

	/**
	 * Returns a typed PropertyNode that represents the specified property.
	 * The property is addressed via its full name. The following example:
	 * \code
	 * myObject.myMember.value
	 * \endcode
	 * returns the PropertyNode of the property member "value"
	 * of the member "myMember" of the object "myObject" that was added via addObject.
	 * Returns nullptr if no property with the given name exists.
	 * @throws XBadCast If the specified property is not of type T.
	 */
	template<typename T>
	const TypedPropertyNode<T>* getPropertyNode(const std::string& name) const
	{
		const PropertyNode* node = getPropertyNode(name);
		if (node == NULL)
			return NULL;
		if ( node->type() != typeName<T>() )
			MIRA_THROW(XBadCast, "The requested type '"
					<< typeName<T>() << "' for property '" << name
					<< "' does not match its real type '"
					<< node->type() << "'");
		return static_cast<const TypedPropertyNode<T>*>(node);
	}

	/**
	 * @copydoc getPropertyNode(const std::string& name) const
	 */
	template<typename T>
	TypedPropertyNode<T>* getPropertyNode(const std::string& name)
	{
		const PropertyManager* This = this;
		return const_cast<TypedPropertyNode<T>*>(This->getPropertyNode<T>(name));
	}

private:

	boost::scoped_ptr<PropertyNode> mRoot;

};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
