/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCManager.h
 *    Contains internal RPCManager class.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/11/08
 */

#ifndef _MIRA_RPCMANAGER_H_
#define _MIRA_RPCMANAGER_H_

#include <set>

#include <platform/Platform.h>

#include <rpc/RPCServer.h>
#include <rpc/RPCClient.h>
#include <rpc/RPCPatternCheck.h>
#include <rpc/RPCCallDefinition.h>
#include <rpc/AbstractDeferredInvoker.h>
#include <rpc/AbstractRPCHandler.h>
#include <rpc/AbstractInterfaceCallbackHandler.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup RPCModule
 * This class is for internal use only.
 * Be aware, that the interface may change significantly in future versions.
 *
 * <pre>
 * Invokation of a remote service:
 *                                                      remote         remote
 *                RPCManager                           RPCManager      service
 *                   |                                     |              |
 *   call() -------->|                 handleRemoteRequest |              |
 *                   |------------------------------------>| invoke       |
 *                   |                                     |------------->|
 *                   | handleRemoteResponse                |<-------------|
 * result via future |<------------------------------------|              |
 *          <--------|                                     |              |
 *
 *
 * Invokation of a local service:
 *
 *                RPCManager                          local service
 *                   |                                      |
 *   call() -------->| handleLocalBinaryRequest      invoke |
 *                   |------------------------------------->|
 * result via future |<-------------------------------------|
 *          <--------|                                      |
 *
 *
 * </pre>
 */
class MIRA_BASE_EXPORT RPCManager : protected DeferredInvokerFinishHandler
{
public:

	/**
	 * Handler that must be implemented by the remote module to send
	 * RPC requests to a remote server.
	 * The handler is specified in the addRemoteService(). Whenever, a
	 * method of the registered remote service is called, the associated handler
	 * is invoked by calling its onRPCrequested() method, where the request
	 * is passed as parameter.
	 * The handler must ship the request to the remote server side. There
	 * the handleRemoteRequest() method of the remote RPCManager will be
	 * called.
	 */
	class RemoteRequestHandler
	{
	public:
		virtual ~RemoteRequestHandler() {}
		virtual void onRPCrequested(Buffer<uint8>&& request) = 0;
	};
	typedef boost::shared_ptr<RemoteRequestHandler> RemoteRequestHandlerPtr;


	/**
	 * Handler that must be implemented by the remote module to send
	 * RPC responses to a remote server which sent a previous RPC request.
	 * The handler is specified in the handleRemoteRequest(), where the
	 * request of an remote RPCManager is invoked. (the request which was
	 * sent by the remote RPCManager using the above RemoteRequestHandler on
	 * the "other side").
	 * The handler is called when the RPC request finishes that was invoked
	 * by the corresponding call of handleRemoteRequest().
	 * The handler must ship the response to the remote server side. There
	 * the handleRemoteResponse() method of the remote RPCManager will be
	 * called.
	 */
	class RemoteFinishHandler
	{
	public:
		virtual ~RemoteFinishHandler() {}
		virtual void onRPCfinished(Buffer<uint8>&& response) = 0;
	};
	typedef boost::shared_ptr<RemoteFinishHandler> RemoteFinishHandlerPtr;

public:

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IRPCManager");
		r.method("getLocalServices", &RPCManager::getLocalServices, this,
		         "Get all registered local services");
		r.method("getLocalService", &RPCManager::getLocalService, this,
		         "Returns a reference to the local service object with the given name",
		         "name", "name of the service", "MyService");
		r.method("getRemoteServices", &RPCManager::getRemoteServices, this,
		         "Get all registered remote services");
		r.method("getRemoteService", &RPCManager::getRemoteService, this,
		         "Returns a reference to the remote service object with the given name",
		         "name", "name of the service", "MyService");
		r.method("existsService", &RPCManager::existsService, this,
		         "Returns true, if a service with the given name exists, otherwise false",
		         "name", "name of the service", "MyService");
		r.method("queryServicesForInterface", &RPCManager::queryServicesForInterface, this,
		         "Returns a string list with the names of all registered services (local"
		         "and remote), that implement the specified interface",
		         "name", "name of the interface", "MyInterface");
		r.method("implementsInterface", &RPCManager::implementsInterface, this,
		         "Returns if the given service implements a certain interface",
		         "name", "name of the service", "MyService",
		         "interface", "name of the interface", "MyInterface");
	}

public: // local service registering

	/**
	 * Adds a local service.
	 * The specified handler is used to process the actual RPC call. It usually
	 * is part of a dispatcher thread that belongs to an authority. Hence, the
	 * call is handled within the authorities thread that registered the service.
	 */
	template <typename Service>
	void addLocalService(const std::string& name, Service& service,
	                     AbstractRPCHandlerPtr handler)
	{
		boost::mutex::scoped_lock lock(mServiceManagementMutex);

		// reflect the service to extend the service object and get the new methods
		// and interfaces
		std::set<RPCSignature> addedMethods;
		std::set<std::string> addedInterfaces;
		const RPCServer::Service& s = mServer.registerServiceObject(name,
				service, &addedMethods, &addedInterfaces);


		// add the service methods and their handlers
		foreach(const RPCSignature& s, addedMethods)
		{
			std::string method = name + "." + s.name;
			auto it = mLocalServiceMethods.find(method);
			if(it == mLocalServiceMethods.end())
				mLocalServiceMethods.insert(std::make_pair(method, handler));
			else {
				if(it->second != handler)
					MIRA_THROW(XLogical, "The service method '" << method <<
					           "' was already registered with a different handler");
			}
		}

		// loop through all new interfaces of the service s
		foreach(const std::string& ifc, addedInterfaces)
			handleCallbacks(ifc, name);

		mLocalServices.insert(name);
	}

	/**
	 * Remove the local service.
	 */
	void removeLocalService(const std::string& name);

	/**
	 * Get all registered local services.
	 */
	std::set<std::string> getLocalServices() const;

	/**
	 * Returns a reference to the local service object with the given name.
	 *
	 * @throw XInvalidParameter If no such service exists.
	 */
	const RPCServer::Service& getLocalService(const std::string& name) const {
		return mServer.getService(name);
	}

public: // remote service registering

	typedef RPCServer::ServiceInfo<RPCServer::MethodInfoSet> RemoteService;

	/**
	 * Adds a remote service.
	 * The specified handler is called, whenever a method of the registered
	 * service is requested. The handler must ship the request to the remote
	 * side, where it is passed to the remote RPCManager via
	 * handleRemoteRequest().
	 * When registering the service, its implemented interfaces need to be
	 * specified.
	 */
	void addRemoteService(const RemoteService& serviceInfo,
	                      RemoteRequestHandlerPtr handler,
	                      uint8 binaryFormatVersion = BinaryBufferSerializer::getSerializerFormatVersion());

	/**
	 * Remove a remote service.
	 */
	void removeRemoteService(const std::string& name);

	/**
	 * Get all registered remote services.
	 */
	std::set<std::string> getRemoteServices() const;

	/**
	 * Returns a reference to the remote service info object with the given name.
	 *
	 * @throw XInvalidParameter If no such service exists.
	 */
	const RemoteService& getRemoteService(const std::string& name) const;

public: // interface handling

	/**
	 * Registers a callback handler that is called whenever a service
	 * registers the interface, which is specified as first parameter.
	 * When a new service is registered that implements the interface the handler
	 * is notified with the name of the service and the interface.
	 */
	void registerCallback(const std::string& interface,
	                      AbstractInterfaceCallbackHandlerPtr callback);

	/**
	 * Unregisters a callback handler for a specific interface.
	 */
	void unregisterCallback(const std::string& interface,
	                        AbstractInterfaceCallbackHandlerPtr callback);

	/**
	 * Unregisters a callback handler for all interfaces.
	 */
	void unregisterCallback(AbstractInterfaceCallbackHandlerPtr callback);

	/**
	 * Returns true, if a service with the given name exists, otherwise false.
	 */
	bool existsService(const std::string& name) const
	{
		boost::mutex::scoped_lock lock(mServiceManagementMutex);

		// check local services first
		if(mServer.existsService(name))
			return true;

		// otherwise examine all remote services
		for(auto it=mRemoteServices.begin(); it!=mRemoteServices.end(); ++it)
			if(it->second.name==name)
				return true; // found a remote service with the desired name

		return false; // nothing found
	}

	/**
	 * Waits until a service with the specified name becomes available.
	 * If a timeout is specified, the method times out if no such service
	 * becomes available after the given time.
	 * Returns true, if the desired services became available, or false, if
	 * the method timed out or the thread was interrupted.
	 */
	bool waitForService(const std::string& name,
	                    Duration timeout = Duration::invalid()) const;

	/**
	 * Waits until a service with the specified interface becomes available.
	 * If a timeout is specified, the method times out if no such service
	 * becomes available after the given time.
	 * Returns the name of the service, if the desired services became
	 * available, or an empty string, if the method timed out or the thread
	 * was interrupted.
	 */
	std::string waitForServiceInterface(const std::string& interface,
	                                    Duration timeout = Duration::invalid()) const;

	/**
	 * Returns a string list with the names of all registered services (local
	 * and remote), that implement the specified interface.
	 */
	std::list<std::string> queryServicesForInterface(const std::string& interface) const
	{
		boost::mutex::scoped_lock lock(mServiceManagementMutex);

		// get the local interfaces first
		std::list<std::string> res = mServer.queryServicesForInterface(interface);

		// now examine all remote services
		for(auto it=mRemoteServices.begin(); it!=mRemoteServices.end(); ++it)
			// if remote service has the interface ...
			if(it->second.interfaces.count(interface)!=0)
				res.push_back(it->second.name); // ... add it

		return res;
	}

	/**
	 * Returns if the given service implements a certain interface
	 */
	bool implementsInterface(const std::string& name, const std::string& interface) const
	{
		boost::mutex::scoped_lock lock(mServiceManagementMutex);
		// get the local interfaces first
		if (mServer.implementsInterface(name, interface))
			return true;

		// now examine all remote services
		auto it = mRemoteServices.find(name);
		if (it == mRemoteServices.end())
			return false;
		return it->second.interfaces.count(interface) > 0;
	}

public: // for handling remote requests and their corresponding responses

	/**
	 * Request to handle the given remote request, whose content is stored
	 * in the specified buffer. The specified finish handler is called when
	 * the RPC request has finished in order to return the response to the
	 * caller.
	 */
	void handleRemoteRequest(Buffer<uint8> buffer,
	                         RemoteFinishHandlerPtr handler = RemoteFinishHandlerPtr());

	/**
	 * Request to handle the given remote request, whose content is stored
	 * in the specified buffer. The specified finish handler is called when
	 * the RPC request has finished in order to return the response to the
	 * caller. The specified format version is used when encoding binary data,
	 * enabling communication with a framework still using legacy format.
	 */
	void handleRemoteRequest(Buffer<uint8> buffer,
	                         RemoteFinishHandlerPtr handler, uint8 binaryFormatVersion);

	/**
	 * Handles the response on a previous remote request, that was sent through
	 * a RemoteRequestHandler.
	 */
	void handleRemoteResponse(Buffer<uint8> buffer);

	/**
	 * Handles the response on a previous remote request, that was sent through
	 * a RemoteRequestHandler. The specified format version is used when
	 * checking the response matches the original request type.
	 */
	void handleRemoteResponse(Buffer<uint8> buffer, uint8 binaryFormatVersion);

public:

	/**
	 * Method that is used by local C++ clients for requesting an rpc call.
	 * It will internally call handleLocalRequest, if the service is managed
	 * by this RPCManager.
	 * Otherwise it looks for the service within the registered remote
	 * services and calls the onRPCrequested method of the service handler.
	 * If the service is completely unknown an XRPC exception is thrown.
	 */
	template<typename R, typename... ARGS>
	RPCFuture<R> call(const std::string& service, const std::string& method, ARGS&&... args)
	{
		Buffer<uint8> buffer;
		boost::mutex::scoped_lock lock(mServiceManagementMutex);
		if (mLocalServices.count(service) > 0) {
			lock.unlock();
			BinaryRPCBackend::ClientRequest request(&buffer);
			RPCFuture<R> future = mClient.call<BinaryRPCBackend, R>(
					request, service, method, forwardRPCParams(std::forward<ARGS>(args))...);
			handleLocalRequest<BinaryRPCBackend>(std::move(buffer), service);
			return std::move(future);
		}
		else {
			auto it = mRemoteServices.find(service);
			if (it == mRemoteServices.end()) {
				MIRA_THROW(XRPC, "The requested service '" << service << "' is not known."); 
			};

			RPCFuture<R> future;
			if (it->second.binaryFormatVersion == BinaryBufferSerializer::getSerializerFormatVersion()) {
				BinaryRPCBackend::ClientRequest request(&buffer);
				future = mClient.call<BinaryRPCBackend, R>(request, service, method,
				                                           forwardRPCParams(std::forward<ARGS>(args))...);
			}
			else if (it->second.binaryFormatVersion == 0) {
				BinaryRPCBackendLegacy::ClientRequest request(&buffer);
				future = mClient.call<BinaryRPCBackendLegacy, R>(
						request, service, method, forwardRPCParams(std::forward<ARGS>(args))...);
			}
			else {
				MIRA_THROW(XRPC, "Requested service expects binary format version " 
				                 << (int)it->second.binaryFormatVersion        
				                 << ". RPCManager::call() only implemented for versions 0, " 
				                 << (int)BinaryBufferSerializer::getSerializerFormatVersion()); 
			};
			it->second.handler->onRPCrequested(std::move(buffer));
			lock.unlock();
			boost::mutex::scoped_lock lock2(mPendingRequestsMutex);
			mPendingRemoteRequests[future.callId()] = service;
			return std::move(future);
		}
	}

	/**
	 * Invokes a JSON RPC call. jsonCall must follow the  JSON RPC 2.0 protocol.
	 * It must contain:
	 *  - "jsonrpc" with exactly the value "2.0"
	 *  - "id" an id that can be anything (int, string, null) but should be an 
	 *         unique identifier
	 *  - "method" a service-method combination of the service to call
	 *             (e.g. "/MyService.myMethod")
	 *  - "params" Optional. Required only for methods with parameters. 
	 *             An array containing the parameters of the method to call
	 *             (e.g. [1, "Hello World"]). 
	 * \code
	 * {
	 * 		"jsonrpc" : "2.0",
	 * 		"id" : 1,
	 * 		"method" : "/ns/Service.setInt",
	 * 		"params" : [255]
	 * }
	 * \endcode
	 * @return Returns a future that will enclose the JSON RPC response containing
	 *         the result of the call.
	 */
	RPCFuture<JSONRPCResponse> callJSON(const json::Value& jsonCall);

	/**
	 * Provided for convenience. See callJSON above. Will convert jsonCall into
	 * a json::Value and calls callJSON(const json::Value& jsonCall) internally.
	 * @param jsonCall Must contain the JSON RPC request in valid JSON syntax.
	 * @return Returns a future that will enclose the JSON RPC response containing
	 *         the result of the call.
	 */
	RPCFuture<JSONRPCResponse> callJSON(const std::string& jsonCall);

	/**
	 * Invokes a JSON RPC call, on the specified service and method with the given
	 * params by generating a JSON RPC 2.0 conform json request. Call id will be
	 * automatically generated.
	 * @param[in] service The service to call
	 * @param[in] method The method on the service to call
	 * @param[in] params The parameters for the method to call
	 *            (must be a json::Array or json null value (default))
	 * @return Returns a future that will enclose the JSON RPC response containing
	 *         the result of the call.
	 */
	RPCFuture<JSONRPCResponse> callJSON(const std::string& service,
	                                    const std::string& method,
	                                    const json::Value& params = json::Value());

	/**
	 * Provided for convenience. See callJSON above. Will convert params string into
	 * an json::Array and calls callJSON(const std::string& service, const std::string& method,
	 * const json::Value& params) internally.
	 * @param[in] service The service to call
	 * @param[in] method The method on the service to call
	 * @param[in] params The parameters of the method as comma separated strings
	 *            (JSON notation, no enclosing [], empty string if no parameters)
	 * @return Returns a future that will enclose the JSON RPC response containing
	 *         the result of the call.
	 */
	RPCFuture<JSONRPCResponse> callJSON(const std::string& service,
	                                    const std::string& method,
	                                    const std::string& params);

	/**
	 * Provided for convenience. Extracts parameters to call
	 * callJSON(const std::string& service, const std::string& method,
	 * const std::string& params) internally.
	 * @param[in] service The service to call
	 * @return Returns a future that will enclose the JSON RPC response containing
	 *         the result of the call.
	 */
	RPCFuture<JSONRPCResponse> callJSON(const RPCCallDefinition& rpc);

protected:

	/**
	 * Internally doing a JSON RPC call.
	 */
	RPCFuture<JSONRPCResponse> callJSONIntern(const std::string& service,
	                                          const std::string& callID,
	                                          const json::Value& request);

	/**
	 * Internally used, to handle rpc requests for services that are also
	 * maintained by this local RPCManager, taking into account that these
	 * service may be handled from the same thread that sent the request.
	 * In this case the method is called immediately to avoid deadlocks.
	 * For all other local requests the handleRequest method below is called.
	 * The content of the request (method name, parameters, etc)
	 * is contained in the specified buffer.
	 */
	template <typename Backend>
	void handleLocalRequest(typename Backend::ContainerType buffer,
	                        const std::string& service);


	/**
	 * Internally used, to handle incoming rpc requests. The specified buffer
	 * contains all necessary data for the RPC call (service name, method
	 * name, parameters, etc). Moreover, a finishHandler must be specified
	 * whose onRPCfinished method is called, when the RPC request was processed.
	 * The last parameter specifies whether the service request is coming from
	 * a remote side. This flag will be stored within the AbstractRequestContext
	 * that is created internally.
	 */
	template <typename Backend>
	void handleRequest(typename Backend::ContainerType buffer,
	                   RemoteFinishHandlerPtr finishHandler, bool remote);

private:

	/**
	 * Calls all service method callbacks that are registered on the specified
	 * interface. As parameters for each callback the interface name and the
	 * specified servicename parameter will be passed.
	 */
	void handleCallbacks(const std::string& interface,
	                     const std::string& servicename);

protected:

	// implements DeferredInvokerFinishHandler
	virtual void onRPCfinished(AbstractDeferredInvoker* invoker);

private:

	/**
	 * Abstract base class for the RequestContext class that stores
	 * additional information with each pending RPC request, e.g.
	 * the finish handler that is called when the request was processed.
	 */
	struct AbstractRequestContext
	{
		AbstractRequestContext(RemoteFinishHandlerPtr iHandler, bool iRemote) :
			handler(iHandler), remote(iRemote){}

		virtual ~AbstractRequestContext() {}

		virtual void onFinished(RPCClient& client) = 0;

	public:
		RemoteFinishHandlerPtr handler;
		bool remote;
	};
	typedef boost::shared_ptr<AbstractRequestContext> AbstractRequestContextPtr;

	/**
	 * Implements AbstractRequestContext to store additional information
	 * for each pending RPC request. The type of that content depends on the
	 * used RPC Backend (Binary, or JSON, etc).
	 * This class is defined and implemented in RPCManager.C.
	 */
	template<typename Backend>
	struct RequestContext;

private:

	// everything is just passed through except const char* (string literal),
	// which is explicitly converted to std::string
	// thus we get the correct RPC signature without the need for explicit cast as in
	// call<void>("Service", "method", std::string("param"));

	template<typename T>
	T&& forwardRPCParams(T&& p) { return std::forward<T>(p); }

	std::string forwardRPCParams(const char* p) { return std::string(p); }

private:

	/// The RPCServer object for managing incoming RPC requests
	RPCServer mServer;

	/// The RPCClient object for generating outgoing RPC requests
	RPCClient mClient;

	/**
	 * Stores additional information for a registered remote service, such
	 * as the RemoteRequestHandler that is called when an outgoing RPC
	 * request is sent.
	 */
	struct RemoteServiceDetail : public RemoteService
	{
		RemoteServiceDetail(const RemoteService& serviceInfo,
		                    RemoteRequestHandlerPtr iHandler,
		                    uint8 iBinaryFormatVersion = BinaryBufferSerializer::getSerializerFormatVersion()) :
			RemoteService(serviceInfo), handler(iHandler),
			binaryFormatVersion(iBinaryFormatVersion) {}

		RemoteRequestHandlerPtr handler;
		uint8 binaryFormatVersion;
	};

	/// mutex to protect the two maps below
	mutable boost::mutex mServiceManagementMutex;

	/**
	 * Contains mapping between all registered local service methods to the
	 * handler that processes each of them.
	 */
	std::map<std::string, AbstractRPCHandlerPtr> mLocalServiceMethods;

	/// Contains the registered local services running in this process
	std::set<std::string> mLocalServices;

	/// map that contains all registered remote services
	std::map<std::string, RemoteServiceDetail> mRemoteServices;

	/// mutex that protects the map below
	mutable boost::mutex mPendingRequestsMutex;
	/// contains all requests that were issued (to a local handler), but where a response is still pending.
	std::map<AbstractDeferredInvoker*, AbstractRequestContextPtr> mPendingRequests;
	/// contains call ID and service name for all requests sent to a remote handler, with response pending.
	std::map<std::string, std::string> mPendingRemoteRequests;

	/// mutex that protects the map below
	mutable boost::mutex mCallbacksMutex;
	/// map of registered callback handlers that get notified when new services are registered
	std::multimap<std::string, AbstractInterfaceCallbackHandlerPtr> mInterfaceCallbacks;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
