/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCInvoker.h
 *    Implements the RPCInvoker and TRPCInvoker classes.
 *
 * @author Erik Einhorn
 * @date   2010/11/07
 */

#ifndef _MIRA_RPCINVOKER_H__
#define _MIRA_RPCINVOKER_H__

#include <functional>
#include <type_traits>
#ifndef Q_MOC_RUN
#include <boost/preprocessor/repetition.hpp>
#endif

#include <error/Exception.h>

#include <rpc/RPCError.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup RPCModule
 *
 * Base of all TRPCInvoker classes which are templates to support different
 * RPC backends.
 */
class RPCInvoker
{
public:
	virtual ~RPCInvoker() {}
};

/**
 * @ingroup RPCModule
 *
 * Invoker that is used to invoke an RPC method call for a special backend.
 * The concrete implementations of the invokers are generated by boost::preprocessor
 * and the macros at the end of this file.
 * For each RPC method that is reflected in the service object an RPCInvoker
 * object is created. The function is as follows:
 * -# Upon construction of the RPCInvoker object the method pointer and the
 *    pointer to the object (for non-static methods) are stored in the constructor
 * -# When calling invoke they RPCInvokers extracts the parameters from the
 *    specified request, uses boost::bind to build an unary function from the
 *    method stored in step 1, and invokes the function. Finally, the result
 *    of the call is stored in the response.
 *
 * @see @ref ServiceRPCPage
 */
template<typename Backend>
class TRPCInvoker : public RPCInvoker
{
public:
	/**
	 * object is pointer to service object whose method is called
	 */
	virtual void invoke(typename Backend::ServerRequest& request,
	                    typename Backend::ServerResponse& response) = 0;
};

//////////////////////////////////////////////////////////////////////////////

namespace Private {

template<typename P>
using StrippedType = typename std::remove_const<typename std::remove_reference<P>::type>::type;

template<typename Backend, typename T>
StrippedType<T> getFromRequest(typename Backend::ServerRequest& request)
{
	StrippedType<T> p;
	request.getParameter(p);
	return p;
}

// helper function to avoid code duplication. must only be called in a catch()-clause!
template<typename Response>
void throwRPCException(Response& response)
{
	try {
		throw;
	}
	catch (SerializableException& ex) {
		response.returnException(RPC_EXCEPTION_IN_CALL, ex);
		return;
	}
	catch (Exception& ex) {
		response.returnException(RPC_EXCEPTION_IN_CALL, ex.what(), ex.callStack());
		return;
	}
	catch (std::exception& ex) {
		response.returnException(RPC_EXCEPTION_IN_CALL, ex.what());
		return;
	}
	catch (...) {
		response.returnException(RPC_EXCEPTION_IN_CALL, "Unknown exception");
		return;
	}
}

// The helper is required since we have methods with and without return value.
// For non-void return types it serializes the return value. There is a specialized version for a return type
// of void which just calls the method without taking care of the return value (which is just void).
template<typename R>
struct RPCInvokeHelper
{
	template<typename Backend, typename Fn, typename... ARGS>
	static void invoke(typename Backend::ServerResponse& response, const Fn& fn, ARGS&&... parameters)
	{
		try {
			response.template returnResult<R>(fn(std::forward<ARGS>(parameters)...));
		}
		catch (...) {
			Private::throwRPCException(response);
		}
	}
};

// Specialization for return type "void"
template<>
struct RPCInvokeHelper<void>
{
	template<typename Backend, typename Fn, typename... ARGS>
	static void invoke(typename Backend::ServerResponse& response, const Fn& fn, ARGS&&... parameters)
	{
		try {
			fn(std::forward<ARGS>(parameters)...);
		}
		catch (...) {
			Private::throwRPCException(response);
			return;
		}
		response.returnVoid();
	}
};

template<typename MemFn, typename Class>
class MemberInvoker
{
public:
	MemberInvoker(const MemFn& func, Class* ptr) : fn{func}, This{ptr} {};
	MemberInvoker(MemFn&& func, Class* ptr) : fn{std::move(func)}, This{ptr} {};

	typedef typename MemFn::result_type ResultType;

	template<typename... ARGS>
	ResultType operator()(ARGS&&... args) const
	{
		return fn(This, std::forward<ARGS>(args)...);
	}

private:
	MemFn fn;
	Class* This;
};

template<typename Backend, typename Function, typename R, typename... ARGS>
struct ConcreteRPCInvoker : public TRPCInvoker<Backend>
{
	template<class P>
	static Private::StrippedType<P> getParameter(typename Backend::ServerRequest& request)
	{
		return Private::getFromRequest<Backend, P>(request);
	}

	ConcreteRPCInvoker(const Function& f) : fn(f) {}
	ConcreteRPCInvoker(Function&& f) : fn(std::move(f)) {}
	void invoke(typename Backend::ServerRequest& request, typename Backend::ServerResponse& response) final
	{
		parseArguments<ARGS...>(request, response);
	}

	// extract parameters recursively:
	// pack of (explicit) template parameters shrinks while pack of function parameters grows
	template<typename Head, typename... Tail, typename... Args>
	void parseArguments(typename Backend::ServerRequest& request, typename Backend::ServerResponse& response,
	                    Args&&... args)
	{
		parseArguments<Tail...>(request, response, std::forward<Args>(args)..., getParameter<Head>(request));
	}

	// stop recursing when all parameters have been extracted from the request
	template<typename... Ignore> // make this a template method (but not an explicit specialization!)
	void parseArguments(typename Backend::ServerRequest& request, typename Backend::ServerResponse& response,
	                    Private::StrippedType<ARGS>&&... args)
	{
		Private::RPCInvokeHelper<R>::template invoke<Backend>(response, fn, std::move(args)...);
	}

	Function fn;
};

} // namespace Private

//////////////////////////////////////////////////////////////////////////////

template<typename Backend, typename R, typename... ARGS, typename Fn>
Private::ConcreteRPCInvoker<Backend,typename std::decay<Fn>::type, R, ARGS...>* make_RPCInvoker(Fn&& f)
{
	return new Private::ConcreteRPCInvoker<Backend,typename std::decay<Fn>::type, R, ARGS...>(
	    std::forward<Fn>(f));
}

//////////////////////////////////////////////////////////////////////////////

} // namespace mira

#endif /* _MIRA_RPCINVOKER_H_ */
