/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AbstractDeferredInvoker.h
 *    Abstract interface for DeferredInvoker.
 *
 * @author Erik Einhorn
 * @date   2010/11/24
 */

#ifndef _MIRA_ABSTRACTDEFERREDINVOKER_H_
#define _MIRA_ABSTRACTDEFERREDINVOKER_H_

#ifndef Q_MOC_RUN
#include <boost/shared_ptr.hpp>
#endif

namespace mira {

//////////////////////////////////////////////////////////////////////////////

// forward decl.
class AbstractDeferredInvoker;

/**
 * @ingroup RPCModule
 *
 * Handler that is called when a deferred RPC call was executed and finished
 * and therefore when the response is available.
 */
class DeferredInvokerFinishHandler
{
public:
	virtual ~DeferredInvokerFinishHandler() {}

	/// called upon finish of the RPC call, the ID of the call is passed as parameter.
	virtual void onRPCfinished(AbstractDeferredInvoker* invoker) = 0;
};

/**
 * Abstract interface for DeferredInvoker which is a class to support
 * different RPC backends.
 */
class AbstractDeferredInvoker
{
public:

	AbstractDeferredInvoker(const std::string& callId,
			                const std::string& service, const std::string& method) :
		mCallId(callId), mServiceName(service), mMethodName(method),
		mFinishHandler(NULL) {}
	virtual ~AbstractDeferredInvoker() {}

public:

	/// Returns the unique ID of the RPC call that is about to be invoked by this invoker.
	const std::string& getCallId() const { return mCallId; }

	/**
	 * Sets a handler that is called after the RPC call was invoked and has finished.
	 */
	void setFinishHandler(DeferredInvokerFinishHandler* handler) {
		mFinishHandler = handler;
	}

	/// Invokes the RPC call that is represented by this DeferredInvoker.
	virtual void invoke() = 0;

	/// Returns the name of the service where the method is invoked.
	const std::string& getServiceName() const { return mServiceName; }

	/// Returns the name of the method that is invoked.
	const std::string& getMethodName() const { return mMethodName; }

protected:

	std::string mCallId;
	std::string mServiceName;
	std::string mMethodName;
	DeferredInvokerFinishHandler* mFinishHandler;
};

typedef boost::shared_ptr<AbstractDeferredInvoker> AbstractDeferredInvokerPtr;

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_ABSTRACTDEFERREDINVOKER_H_ */
