/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Memory.h
 *    System independent way of getting the memory usage of a process.
 *
 * @author Tim Langner
 * @date   2011/11/27
 */

#ifndef _MIRA_MEMORY_H_
#define _MIRA_MEMORY_H_

#include <string>
#include <list>

#include <platform/Types.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Memory usage information in kB.
 * If a certain measure is not supported on the operating system, the
 * corresponding value remains 0.
 */
struct MemoryUsage
{
	MemoryUsage() :
		virtualSize(0),
		residentSetSize(0),
		totalHeap(0),
		totalStack(0){}

	/**
	 * The total size of virtual memory used by this process. However, due to
	 * memory virtualization, this value usually is meaningless for a process.
	 */
	uint64 virtualSize;

	/**
	 * The resident/working set size, i.e. the physical memory that is occupied
	 * by the process. Note that this includes memory for shared libraries
	 * that are also used by other processes. Therefore, this value usually
	 * overestimates the real memory usage of a process.
	 * To get the amount of allocated data used by the process use the
	 * totalHeap and totalStack stack member.
	 */
	uint64 residentSetSize;

	/**
	 * The total amount of used heap memory allocated by this process.
	 * When summed with totalStack the resulting value indicates the memory
	 * that is allocated by the users program code and can be used to detect
	 * memory leaks.
	 */
	uint64 totalHeap;

	/**
	 * The total stack size of all threads.
	 * When summed with totalHeap the resulting value indicates the memory
	 * that is allocated by the users program code and can be used to detect
	 * memory leaks.
	 */
	uint64 totalStack;
};

/**
 * Returns information about the memory used currently by this process.
 */
MIRA_BASE_EXPORT MemoryUsage getMemoryUsage();


/**
 * Detailed information of used memory within a memory block.
 */
struct MemoryBlockInfo
{
	MemoryBlockInfo() :
		size(0), totalPrivate(0), totalShared(0),
		swap(0), rss(0), pss(0){}

	/// The name of this block
	std::string name;

	/// The address or address range of this block
	std::string address;

	/// The total size of this memory block
	uint64 size;
	/// The amount of memory in this block used by this process only (clean and dirty pages)
	uint64 totalPrivate;
	/// The amount of memory in this block shared among other processes (clean and dirty pages)
	uint64 totalShared;
	/// The amount of memory that was swapped
	uint64 swap;
	/// The resident set size (including shared memory)
	uint64 rss;
	/// The proportional set size
	uint64 pss;
};

typedef std::list<MemoryBlockInfo> MemoryBlockInfoList;

/**
 * Returns detailed information on memory blocks used by this process.
 * Currently supported on Linux operating systems only.
 */
MIRA_BASE_EXPORT MemoryBlockInfoList getDetailedMemoryBlockUsage();


///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
