/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Random.h
 *    Functions for generating random numbers
 *
 * @author Tim Langner
 * @date   2011/07/15
 */

#ifndef _MIRA_RANDOM_H_
#define _MIRA_RANDOM_H_

#include <stdlib.h>

#include <utils/Singleton.h>
#include <utils/Time.h>

#define MIRA_RANDOM mira::RandomGeneratorSingleton::instance()

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

/**
 * Singleton class for generating random numbers.
 * Use the MIRA_RANDOM macro to access the singleton.
 *
 * Usage:
 * \code
 * #include <math/Random.h>
 * ...
 *
 * // sample uniform random number between 10 and 30
 * int rndnumber1 = MIRA_RANDOM.uniform(10,30);
 *
 * // sample normal distributed random number with sigma=2
 * float rndnumber2 = MIRA_RANDOM.normal(2.0);
 *
 * \endcode
 *
 * @ingroup MathModule
 */
class RandomGeneratorSingleton : public LazySingleton<RandomGeneratorSingleton>
{
private:
	friend struct mira::singleton::CreateUsingNew<RandomGeneratorSingleton>;
	RandomGeneratorSingleton() {
		seed();
	}

public:
	/**
	 * Seed the random number generator.
	 * Is done automatically when creating the singleton
	 * @param[in] seed The seed value: Default = 0 uses the current time for
	 *                 seeding the random generator. Therefore, the result will
	 *                 be a non-reproducible pseudo-random sequence.
	 *                 All other values allow to seed the random generator with
	 *                 a specific value and will result in a reproducible
	 *                 pseudo-random sequence.
	 */
	void seed(uint64 seed = 0)
	{
#if defined(MIRA_LINUX)
		if (seed == 0)
			srand48(Time::now().toUnixTimestamp());
		else
			srand48(seed);
#elif defined(MIRA_WINDOWS)
		if (seed == 0)
			srand((unsigned int)(Time::now().toUnixTimestamp()));
		else
			srand((unsigned int)seed);
#endif
	}

	/**
	 * Platform independent generation of random, double precision
	 * floating point random numbers in the interval [0.0, 1.0)
	 */
	double random()
	{
#if defined(MIRA_LINUX)
		return drand48();
#elif defined(MIRA_WINDOWS)
		return 1.0*rand()/RAND_MAX;
#endif
	}

	/**
	 * Generate uniform distributed random numbers in the
	 * interval [min, max)
	 */
	template<typename T>
	T uniform(T min, T max)
	{
		return min + (T)(random()*(max-min));
	}

	/**
	 * Generate standard normally distributed random numbers
	 * using a source of uniformly distributed random numbers.
	 * Sigma must be >= 0.0!
	 */
	double normal(double sigma = 1.0)
	{
		assert(sigma >= 0.0);
		if (sigma == 0.0)
			return 0.0;
		double x, y, r2;
		do {
			// choose x,y in uniform square [-1,-1) to [+1,+1)
			x = uniform(-1.0, 1.0);
			y = uniform(-1.0, 1.0);

			// see if it is in the unit circle
			r2 = x*x + y*y;
		} while ((r2 > 1.0) || (r2 == 0));

		// Marsaglia polar method
		return sigma * y * sqrt(-2.0*log(r2)/r2);
	}

	/// deprecated, use normal() instead.
	double gaussian(double sigma = 1.0) {
		return normal(sigma);
	}

};

///////////////////////////////////////////////////////////////////////////////

}

#endif
