/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file NearestNeighborInterpolator.h
 *    Filter that returns the nearest neighbor.
 *
 * @author Erik Einhorn
 * @date   2010/10/03
 */

#ifndef _MIRA_NEARESTNEIGHBORINTERPOLATOR_H__
#define _MIRA_NEARESTNEIGHBORINTERPOLATOR_H__

#include <assert.h>

#include <math/IntervalFilter.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * 1D nearest neighbor interpolator.
 * From two samples that are provided in the apply() method, this Filter will
 * always return the value that is closest to the specified value xx.
 *
 * If both samples have the same distance from xx, the second value will
 * be returned.
 *
 * @ingroup MathModule
 */
class NearestNeighborInterpolator : public IntervalFilter
{
public:
	bool canExtrapolate() const { return true; }
	int samples() const         { return 2; }
	int samplesBefore() const   { return 1; }
	int samplesAfter() const    { return 1; }

	template <typename Tx, typename Ty, typename ContainerTx, typename ContainerTy>
	const Ty& apply(const ContainerTx& x, const ContainerTy& y, const Tx& xx)
	{
		assert(x.size()==2);
		assert(x.size()==y.size());

		// catch extrapolation cases
		if(xx <= x.front())
			return y.front();

		if(xx >= x.back())
			return y.back();

		// find the closet point of the two points
		if(xx-x.front() < x.back()-xx)
			return y.front();
		else
			return y.back();
	}
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_NEARESTNEIGHBORINTERPOLATOR_H_ */
