/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Line.h
 *    A class for a n-dimensional line.
 *
 * @author Erik Einhorn
 * @date   2011/05/23
 */

#ifndef _MIRA_LINE_H_
#define _MIRA_LINE_H_

#include <geometry/Point.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Represents a line segment that is spanned by two given points.
 */
template <typename T, int D>
class Line
{
public:
	typedef Point<T,D> PointType;

public:
	/// Creates an uninitialized line segment
	Line() {}

	/// Creates a line segment that is spanned by two given points
	Line(const PointType& p1, const PointType& p2) :
		first(p1), second(p2) {}

public:
	/// Returns a reference to the first point of the line segment
	const PointType& front() const { return first; }

	/// Returns a reference to the first point of the line segment
	PointType& front() { return first; }

	/// Returns a reference to the second point of the line segment
	const PointType& back() const { return second; }

	/// Returns a reference to the second point of the line segment
	PointType& back() { return second; }

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("P0", first,  "The first point");
		r.property("P1", second, "The second point");
	}

public:
	/// The two points of the line segment
	PointType first, second; // must be called first and second
	                         // for compatibility with boost::geometry

};

///////////////////////////////////////////////////////////////////////////////

typedef Line<int, 2>   Line2i; ///< A 2D integer line.
typedef Line<int, 3>   Line3i; ///< A 3D integer line.
typedef Line<float, 2> Line2f; ///< A 2D 32 bit floating precision line.
typedef Line<float, 3> Line3f; ///< A 3D 32 bit floating precision line.

///////////////////////////////////////////////////////////////////////////////

} // end of MIRA namespace

///@cond INTERNAL

// BOOST specialization on Line to accept them as geometry entity
namespace boost { namespace geometry { namespace traits {

///////////////////////////////////////////////////////////////////////////////

template <typename T, int D>
struct tag<mira::Line<T,D> > {
	typedef segment_tag type;
};

template <typename T, int D>
struct point_type<mira::Line<T,D> > {
	typedef mira::Point<T,D> type;
};

template <typename T, int D, std::size_t Dimension>
struct indexed_access<mira::Line<T,D>, 0, Dimension>
{
	typedef mira::Line<T,D> segment_type;
	typedef typename geometry::coordinate_type<segment_type>::type coordinate_type;

	static inline coordinate_type get(segment_type const& s) {
		return geometry::get<Dimension>(s.first);
	}

	static inline void set(segment_type& s, coordinate_type const& value) {
		geometry::set<Dimension>(s.first, value);
	}
};

template <typename T, int D, std::size_t Dimension>
struct indexed_access<mira::Line<T,D>, 1, Dimension>
{
	typedef mira::Line<T,D> segment_type;
	typedef typename geometry::coordinate_type<segment_type>::type coordinate_type;

	static inline coordinate_type get(segment_type const& s) {
		return geometry::get<Dimension>(s.second);
	}

	static inline void set(segment_type& s, coordinate_type const& value) {
		geometry::set<Dimension>(s.second, value);
	}
};

//////////////////////////////////////////////////////////////////////////////

}}} // end of boost namespaces

///@endcond

//////////////////////////////////////////////////////////////////////////////

#endif
